##START#OF#EXPORT_ASM_PMWIN##################################################
# geossav_export_asm_pmwin.tcl
#
# Source script creating the export_asm_pmwin widget in GEOSSAV, and running 
# the export of data. , The output files can be used in ASM and PMWIN.
# The called c/c++ procs are in the geossav.c file.
# Developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the EXPORT_ASM_PMWIN
#
global exedir
source $exedir/Geossav_export_asm_pmwin_globals.tcl

set asmpar(datafl) "*.out"                     ;# simulated realizations file, simplified Geo-EAS format
set asmpar(nsim) 1                             ;# number of simulations to consider from simulation data
set asmpar(max_nsim) 1                         ;# maximum number of simulations to consider from simulation data
set asmpar(vartype) 0                          ;# variable type: 1=continuous (cdf), 0=categorical (pdf)
set asmpar(nx) 100                             ;# number of grid cells (x axis)
set asmpar(xsiz) 10                            ;# size of grid cell in x-direction
set asmpar(ny) 100                             ;# number of grid cells (y axis)
set asmpar(xsiz) 10                            ;# size of grid cell in y-direction
set asmpar(nz) 1                               ;# number of grid cells (z axis)
set asmpar(xsiz) 10                            ;# size of grid cell in z-direction
set asmpar(charfl) "*.gtd"                     ;# ground type data file, simplified Geo-EAS format
set asmpar(mean) 0                             ;# considering mean values from ground type data, 1=yes, 0=no
set asmpar(cmean) 1                            ;# column number for the mean
set asmpar(dist) 1                             ;# considering distribution values from ground type data, 1=yes, 0=no
set asmpar(cdist) 2                            ;# column number for the standard deviation
set asmpar(nsimchar) 1                         ;# number of simulations to consider from characterization data
set asmpar(icoli) 1                            ;# column number for the characterization variables
set asmpar(infofl) "*.out"                     ;# information file to the realization file, simplified Geo-EAS format
set asmpar(ncat) 3                             ;# number of thresholds or categories
set asmpar(cat) 1                              ;# threshold values or category codes
set asmpar(dflbc) 0                            ;# counter for the activation of the datafl browse button
set asmpar(outfl) "*.out"                      ;# output data file/s containing the values for the import in asm
set asmpar(noutfl) 1                           ;# number of created ouput files
set asmpar(sname) "( *.out )"                  ;# output file extension modification and explanation
set asmpar(charvar) 0                          ;# number of characterization variables (ground types)
set asmpar(oldcharvar) 0                       ;# old number of characterization variables (ground types)
                                               ;# if simulated realizations data were fed newly
set asmpar(char_parameters) 0                  ;# number of characterization variables
set asmpar(cancel_counter_characterization) 0  ;# counter for the exit from the EXPORT_ASM_PMWIN parameters tabnotebook
                                               ;# if characterization data were fed

#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc export_asm_read_char_data { fname } {
    global asmpar
    global dim_name_char

    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_char $dum
	    set tmp_parameters $i
	}
	if { $asmpar(cancel_counter_characterization) == 0 } {
	    set asmpar(charvar) $tmp_parameters
	} elseif { $asmpar(cancel_counter_characterization) != 0 } {
	    set asmpar(charvar) $asmpar(charvar)
	}
	close $inf
	set asmpar(char_parameters) $tmp_parameters
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc check_number_nsim_sim { entry newstr char } {
    global asmpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if { $asmpar(datafl) == {*.out} } {
	if {$current < 100} {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif {$current >= 100 && [string match {[0-9]} $char]} {
	    $entry delete 0 end
	    $entry insert end "1000"
	}
    } elseif { $asmpar(datafl) != {*.out} } {
	if { $newstr < $asmpar(max_nsim) } {
	    return [regexp {^[1-9]*$} $newstr]
	} elseif { $newstr >= $asmpar(max_nsim) } {
	    $entry delete 0 end  
	    $entry insert end "$asmpar(max_nsim)"
	}
    }
}
#============================================================================


#############################################################################
proc check_number_nsim_char { entry newstr char } {
    global asmpar

    set current [$entry get]
    catch {scan $current {%d} current}

    if {$current < 10} {
	return [regexp {^[1-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc number_output_files { } {
    global asmpar
    
    catch {scan $asmpar(nsim) {%d} asmpar(nsim)}
    catch {scan $asmpar(nz) {%d} asmpar(nz)}
    catch {scan $asmpar(nsimchar) {%d} asmpar(nsimchar)}

    
    if { $asmpar(vartype) == "Categorical" } {
	if { $asmpar(mean) == 0 } {
	    if { $asmpar(nsim) == 0 || $asmpar(nsimchar) == 0 || $asmpar(nsim) == {} || $asmpar(nsimchar) == {} } {
		set asmpar(noutfl) "?"
		set asmpar(sname) "?"
	    } else {
		if { $asmpar(datafl) == {*.out} } {
		    set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nz) * $asmpar(nsimchar)]
		    set asmpar(sname) "( *.out )"
		} elseif { $asmpar(datafl) != {*.out} && $asmpar(charfl) != {*.gtd} } {
		    if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
			set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nsimchar)]
			set asmpar(sname) "( *.out_s1-$asmpar(nsim),_c1-$asmpar(nsimchar) );   'simulation, characterization'"
		    } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
			if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
			    set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nsimchar)]
			set asmpar(sname) "( *.out_s1-$asmpar(nsim),_c1-$asmpar(nsimchar) );   'simulation, characterization'"
			} elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
			    set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nz) * $asmpar(nsimchar)]
			    set asmpar(sname) "( *.out_s1-$asmpar(nsim),_l1-$asmpar(nz),_c1-$asmpar(nsimchar) )   'simulation, layer, characterization'"
			}
		    }
		} elseif { $asmpar(datafl) != {*.out} && $asmpar(charfl) == {*.gtd} } {
		    if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
			set asmpar(noutfl) $asmpar(nsim)
			set asmpar(sname) "( *.out_s1-$asmpar(nsim) );   'simulation'"
		    } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
			if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
			    set asmpar(noutfl) $asmpar(nsim)
			set asmpar(sname) "( *.out_s1-$asmpar(nsim) );   'simulation'"
			} elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
			    set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nz)]
			    set asmpar(sname) "( *.out_s1-$asmpar(nsim),_l1-$asmpar(nz) )   'simulation, layer'"
			}
		    }
		}
	    }
	} elseif { $asmpar(mean) == 1 } {
	    if { $asmpar(nsim) == 0 || $asmpar(nsimchar) == 0 || $asmpar(nsim) == {} || $asmpar(nsimchar) == {} } {
		set asmpar(noutfl) "?"
		set asmpar(sname) "?"
	    } else {
		if { $asmpar(datafl) == {*.out} } {
		    set asmpar(noutfl) $asmpar(nsim)
		    set asmpar(sname) "( *.out )"
		} elseif { $asmpar(datafl) != {*.out} } {
		    if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
			set asmpar(noutfl) $asmpar(nsim)
			set asmpar(sname) "( *.out_s1-$asmpar(nsim),_c1 );   'simulation, characterization'"
		    } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
			if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
			    set asmpar(noutfl) $asmpar(nsim)
			    set asmpar(sname) "( *.out_s1-$asmpar(nsim),_c1 );   'simulation, characterization'"
			} elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
			    set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nz)]
			    set asmpar(sname) "( *.out_s1-$asmpar(nsim),_l1-$asmpar(nz),_c1 )   'simulation, layer, characterization'"
			}
		    }
		}
	    }
	}
    } elseif { $asmpar(vartype) == "Continous" } {
	if { $asmpar(nsim) == 0 || $asmpar(nsimchar) == 0 || $asmpar(nsim) == {} || $asmpar(nsimchar) == {} } {
	    set asmpar(noutfl) "?"
	    set asmpar(sname) "?"
	} else {
	    if { $asmpar(datafl) == {*.out} } {
		set asmpar(noutfl) $asmpar(nsim)
		set asmpar(sname) "( *.out )"
	    } elseif { $asmpar(datafl) != {*.out} } {
		if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
		    set asmpar(noutfl) $asmpar(nsim)
		    set asmpar(sname) "( *.out_s1-$asmpar(nsim) );   'simulation'"
		} elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
		    if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
			set asmpar(noutfl) $asmpar(nsim)
			set asmpar(sname) "( *.out_s1-$asmpar(nsim) );   'simulation'"
		    } elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
			set asmpar(noutfl) [expr $asmpar(nsim) * $asmpar(nz)]
			set asmpar(sname) "( *.out_s1-$asmpar(nsim),_l1-$asmpar(nz) )   'simulation, layer'"
		    }
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc mean_value_selection { } {
    global asmpar
    global distribution_value_combo_w
    global realization_char_w
    
    if { $asmpar(mean) == 1 } {
	set asmpar(dist) 0
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $asmpar(mean) == 0 } {
	set asmpar(dist) 1
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$realization_char_w configure -state normal -textbackground white -foreground black
    }
    number_output_files
}
#============================================================================


#############################################################################
proc distribution_value_selection { } {
    global asmpar
    global distribution_value_combo_w
    global realization_char_w

    if { $asmpar(dist) == 1 } {
	set asmpar(mean) 0
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$realization_char_w configure -state normal -textbackground white -foreground black
    } elseif { $asmpar(dist) == 0 } {
	set asmpar(mean) 1
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
    }
    number_output_files
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc read_infofl_params { fname } {
    global asmpar
    global charfl_w
    global charfl_browse_w
    global realization_char_w
    global mean_value_w
    global distribution_value_w
    global mean_value_combo_w
    global distribution_value_combo_w
    global datafl_w
    global char_vals
    global char_table_w
    global dim_name_char
    global tmp_charvar
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	set asmpar(datafl) "*.out"
	$datafl_w configure -textvariable asmpar(datafl)
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\nNo such file or directory."
	return 0
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "information for sisim data export" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Information for SISIM data export" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File content error" \
		-icon error \
		-message "Information to the simulated realizations file can not be found.\n"
	    return 0
	}
    }

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_vartype [format "%d" [string range $line 0 $space]]

    ### reads the nsim: number of realizations
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nsim) [format "%d" [string range $line 0 $space]]
    set asmpar(max_nsim) $asmpar(nsim)

    ### reads the nx, xsiz: number, size of grid cells (x direction)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set asmpar(xsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the ny, ysiz: number, size of grid cells (y direction)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(ny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set asmpar(ysiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the nz, zsiz: number, size of grid cells (z direction)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nz) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set asmpar(zsiz) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### destroys the content of the characterization table
    if { $asmpar(dflbc) == 1 } {
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	}
    } elseif { $asmpar(dflbc) != 1 } {
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	}
    }

    ### reads the ncat: number of thresholds / categories
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(ncat) [format "%d" [string range $line 0 $space]]

    ### setting up the rows of the char table
    $char_table_w configure -rows [expr $asmpar(ncat) + 1]

    ### reads the cat: thresholds / categories and fills part of the char table
    $char_table_w configure -rows [expr $asmpar(ncat) + 1]
    set line [gets $inf]
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	
	set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update

	set space [string first " " $line]
	set asmpar(cat,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
				      -textvariable asmpar(cat,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 19]
	update
	

	if { $asmpar(charfl) == {*.gtd} } {
 	    set asmpar(icoli,$ii) $ii
	} elseif { $asmpar(charfl) != {*.gtd} } {
	    if { $asmpar(oldcharvar) < [expr $asmpar(ncat) + 1] } {
		set asmpar(icoli,$ii) 1
	    } elseif { $asmpar(oldcharvar) >= [expr $asmpar(ncat) + 1] } {
		set asmpar(icoli,$ii) $asmpar(icoli,$ii)
	    }
	}
	set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
				      -textvariable asmpar(icoli,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 29 \
				      -listheight 100]
	$char_vals($ii,combo) clear list
	foreach i $dim_name_char {
	    $char_vals($ii,combo) insert list end $i
	}
	if { $asmpar(charfl) == {*.gtd} } {
	    $char_vals($ii,combo) selection set 0
	    set asmpar(icoli,$ii) [lindex $dim_name_char 0]
	} elseif { $asmpar(charfl) != {*.gtd} } {
	    if { $tmp_vartype == 0 } {
		if { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
		    $char_vals($ii,combo) selection set [expr $ii - 1]
		    set asmpar(icoli,$ii) [lindex $dim_name_char [expr $ii - 1]]
		} elseif { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
		    $char_vals($ii,combo) selection set [expr $ii - 1]
		    set asmpar(icoli,$ii) [lindex $dim_name_char [expr $ii - 1]]
		}
	    } elseif { $tmp_vartype == 1 } {
		$char_vals($ii,combo) selection set 0
		set asmpar(icoli,$ii) [lindex $dim_name_char 0]	    
	    }
	}
	$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	update

	if { $asmpar(charfl) == {*.gtd} } {
	    $char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $asmpar(charfl) != {*.gtd} } {
	    $char_vals($ii,entry) configure -state normal -textbackground white -foreground black
	    $char_vals($ii,combo) configure -state normal -textbackground white -foreground black
	}
	
	$char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	$char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	$char_table_w window configure $ii,2 -window $char_vals($ii,combo)
    }
    
    ### control the selection of the mean value and distribution value comboboxes depending on the content of the char data
    if { $asmpar(charfl) != {*.gtd} } { 
	### mean value combobox
	if { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
	    set tmp_cmean $asmpar(cmean)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_char {
		$mean_value_combo_w insert list end $i
		$mean_value_combo_w delete list [expr $asmpar(charvar) + 1]
		if { $i == $tmp_cmean } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($asmpar(cancel_counter_characterization) * ($asmpar(charvar) + 1))]
	    $mean_value_combo_w selection set $sel
	    set asmpar(cmean) [lindex $dim_name_char $sel]

	    ### distribution value combobox
	    set tmp_cdist $asmpar(cdist)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	    foreach i $dim_name_char {
		$distribution_value_combo_w insert list end $i
		$distribution_value_combo_w delete list [expr $asmpar(charvar) + 1]
		if { $i == $tmp_cdist } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($asmpar(cancel_counter_characterization) * ($asmpar(charvar) + 1))]
	    $distribution_value_combo_w selection set $sel
	    set asmpar(cdist) [lindex $dim_name_char $sel]
	} elseif { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
	    ### mean value combobox
	    $mean_value_combo_w selection set 0
	    set asmpar(cmean) [lindex $dim_name_char 0]
	    
	    ### distribution value combobox
	    $distribution_value_combo_w selection set 0
	    set asmpar(cdist) [lindex $dim_name_char 0]
	}
    }

    ### setting the characterization input widgets normal/disabled depending on the vartype
    if { $tmp_vartype == 1 } {
	$charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
	if { $asmpar(charfl) == {*.gtd} } {
	    for { set ii 1 } { $ii <= 3 } { incr ii } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	} elseif { $asmpar(charfl) != {*.gtd} } {
	    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
	$mean_value_w               configure -state disabled
	$distribution_value_w       configure -state disabled
	$mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50

    } elseif { $tmp_vartype == 0 } {
	if { $asmpar(charfl) == {*.gtd} } {
	    $charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	    $charfl_browse_w configure -state normal
	    for { set ii 1 } { $ii <= 3 } { incr ii } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	    $mean_value_w               configure -state disabled
	    $distribution_value_w       configure -state disabled
	    $mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $asmpar(charfl) != {*.gtd} } {
	    $charfl_w             configure -state normal -textbackground white -foreground black
	    $charfl_browse_w      configure -state normal
	    $mean_value_w         configure -state normal
	    $distribution_value_w configure -state normal
	    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
		$char_vals($ii,entry) configure -state normal -textbackground white -foreground black
	    	$char_vals($ii,combo) configure -state normal -textbackground white -foreground black
	    }
	    if { $asmpar(mean) == 1 } {
		$mean_value_combo_w         configure -state normal -textbackground white -foreground black
		$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
		$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	    } elseif { $asmpar(dist) == 1 } {
		$mean_value_combo_w         configure -state normal -textbackground white -foreground black
		$distribution_value_combo_w configure -state normal -textbackground white -foreground black
		$realization_char_w         configure -state normal -textbackground white -foreground black
	    }
	}
    }

    if { $tmp_vartype == 0 } {
	set asmpar(vartype) "Categorical"
    } elseif { $tmp_vartype == 1 } {
    	set asmpar(vartype) "Continous"
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc read_export_asm_params { fname } {
    global asmpar
    global dim_name_char
    global export_asm_box_w
    global datafl_w
    global realization_sim_w
    global char_table_w
    global char_vals
    global charfl_w
    global charfl_browse_w
    global mean_value_w
    global distribution_value_w
    global mean_value_combo_w
    global realization_char_w
    global distribution_value_combo_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are read
    . configure -cursor watch
    update
    
    ### searches the line "parameters for export_asm" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for EXPORT_ASM_PMWIN" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in export_asm box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $export_asm_box_w "EXPORT ASM, PMWIN parameters   -   $fname_short"

    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### destroys the content of the ivar-table, the vario_dir_table and the vario_type_table 
    if { $asmpar(datafl) == {*.out} } {
	for { set ii 1 } { $ii <= 3 } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	}
    } elseif { $asmpar(datafl) != {*.out} } {
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    destroy $char_vals($ii,label)
	    destroy $char_vals($ii,entry)
	    destroy $char_vals($ii,combo)
	}
    }    
    
    ### reads name of datafl: file with simulated realizations
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set asmpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set asmpar(datafl) [format "%s" $line]
    }
    $datafl_w configure -state normal -textbackground white -foreground black

    ### reads the nsim: number of realizations to consider
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nsim) [format "%d" [string range $line 0 $space]]
    set asmpar(max_nsim) $asmpar(nsim)
    $realization_sim_w configure -state normal -textbackground white -foreground black

    ### reads the vartype: variable type (1=cont.(cdf), 0=cat.(pdf))
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_vartype [format "%d" [string range $line 0 $space]]
    if { $tmp_vartype == 1 } {
	set asmpar(vartype) "Continous"
    } elseif { $tmp_vartype == 0 } {
	set asmpar(vartype) "Categorical"
    }

    ### reads the nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set asmpar(ny) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set asmpar(nz) [format "%d" [string range $line 0 [expr $space - 1]]]

    ### reads name of charfl: file with characterization data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set asmpar(charfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set asmpar(calibfl) $line
    }

    ### reads the ncat: number of thresholds or categories
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(ncat) [format "%d" [string range $line 0 $space]]

    ### reads the cat(i): thresholds or categories
    set line [gets $inf]
    set catline $line
    
    ### reads the icoli(i): column number for the characterization variables
    set line [gets $inf]
    set icoliline $line

    ### fills the char-table
    $char_table_w configure -rows [expr $asmpar(ncat) + 1]

    if { $asmpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_asm_read_char_data $asmpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $asmpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
    }

    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	
	set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update
	
	set space [string first " " $catline]
	set asmpar(cat,$ii) [format "%g" [string range $catline 0 [expr $space - 1]]]
	set catline [string range $catline [expr $space + 1] end]
	set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
				      -textvariable asmpar(cat,$ii) \
				      -validate {regexpvalidate %c} \
				      -state normal \
				      -width 19 \
				      -textbackground white \
				      -foreground black]
	update

	set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
				      -textvariable asmpar(icoli,$ii) \
				      -validate {regexpvalidate %c} \
				      -width 29 \
				      -listheight 100]
	$char_vals($ii,combo) clear list
	foreach i $dim_name_char {
	    $char_vals($ii,combo) insert list end $i
	}
	set space [string first " " $icoliline]
	set asmpar(icoli,$ii) [format "%d" [string range $icoliline 0 [expr $space - 1]]]
	set icoliline [string range $icoliline [expr $space + 1] end]
	$char_vals($ii,combo) selection set $asmpar(icoli,$ii)
	update
	
	$char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	$char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	$char_table_w window configure $ii,2 -window $char_vals($ii,combo)
    }

    ### reads the mean: mean value to consider (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(mean) [format "%d" [string range $line 0 $space]]

    ### reads the dist: value from the distribution to consider (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(dist) [format "%d" [string range $line 0 $space]]

    ### reads the cmean: column number for the mean
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_asmpar(cmean) [format "%g" [string range $line 0 $space]]

    if { $asmpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_asm_read_char_data $asmpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $asmpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
	$mean_value_combo_w configure -state normal -textbackground white -foreground black
	$mean_value_combo_w clear list
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	}
	$mean_value_combo_w selection set $tmp_asmpar(cmean)
    }

    ### reads the cdist: column number for the standard deviation
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_asmpar(cdist) [format "%g" [string range $line 0 $space]]

    if { $asmpar(datafl) != {*.out} } {
	unset dim_name_char
	if { [export_asm_read_char_data $asmpar(charfl)] != 1 } {
	    set dim_name_char {}
	    for { set i 1 } { $i <= $asmpar(charvar) } { incr i } {
		lappend dim_name_data $i
	    }
	}
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	$distribution_value_combo_w clear list
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	}
	$distribution_value_combo_w selection set $tmp_asmpar(cdist)
    }

    ### reads the nsimchar: number of characterization realizations to consider
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(nsimchar) [format "%d" [string range $line 0 $space]]

    ### setting the characterization input widgets normal/disabled depending on the vartype
    if { $tmp_vartype == 1 } {
	$charfl_w        configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    $char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	}
	$mean_value_w               configure -state disabled
	$distribution_value_w       configure -state disabled
	$mean_value_combo_w         configure -state disabled -textbackground gray85 -foreground gray50
	$realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50

    } elseif { $tmp_vartype == 0 } {
	$charfl_w             configure -state normal -textbackground white -foreground black
	$charfl_browse_w      configure -state normal
	$mean_value_w         configure -state normal
	$distribution_value_w configure -state normal
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    $char_vals($ii,entry) configure -state normal -textbackground white -foreground black
	    $char_vals($ii,combo) configure -state normal -textbackground white -foreground black
	}
	if { $asmpar(mean) == 1 } {
	    $mean_value_combo_w         configure -state normal -textbackground white -foreground black
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w         configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $asmpar(dist) == 1 } {
	    $mean_value_combo_w         configure -state normal -textbackground white -foreground black
	    $distribution_value_combo_w configure -state normal -textbackground white -foreground black
	    $realization_char_w         configure -state normal -textbackground white -foreground black
	}
    }

    ### reads the outfl: file for output data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set asmpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set asmpar(outfl) $line
    }

    ### reads the noutfl: number of output files
    set line [gets $inf]
    set space [string first " " $line]
    set asmpar(noutfl) [format "%d" [string range $line 0 $space]]

    ### reads the sname: extension of output files
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set asmpar(sname) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set asmpar(sname) $line
    }

    ### reset the cancel counter
    set asmpar(cancel_counter_characterization) 0

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_export_asm_params { fname } {
    global tcl_platform
    global version
    global asmpar
    global export_asm_box_w
    global variable_type_value_w
    global char_vals
    global mean_value_combo_w
    global distribution_value_combo_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### sets the cursor on watch as long as parameters are written
    . configure -cursor watch
    update
    
    ### set file name in bicalib box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $export_asm_box_w "EXPORT ASM, PMWIN parameters   -   $fname_short"

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for EXPORT_ASM_PMWIN"
    puts $outf "*******************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with simulated realizations
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(datafl)]
    set line [format "%s" $asmpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\simulated realizations"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### nsim: number of realizations to consider
    set comment "\\number of realizations to consider"
    catch {scan [$realization_sim_w get] {%d} asmpar(nsim)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(nsim) $comment]

    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf)
    set comment "\\variable type (1=cont.(cdf), 0=cat.(pdf)"
    set line [format "%s" $asmpar(vartype)] 
    if { $line == "Categorical" } {
	set asmpar(vartype) 0
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(vartype) $comment]
	set asmpar(vartype) "Categorical"
    } elseif { $line == "Continous" } {
	set asmpar(vartype) 1
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(vartype) $comment]
	set asmpar(vartype) "Continous"
    }

    ### nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $asmpar(nx) $asmpar(ny) $asmpar(nz)]
    set line [format "%d %d %d" $asmpar(nx) $asmpar(ny) $asmpar(nz)]
    set line_length [string length $line]
    set comment "\\number of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### charfl: file with characterization data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(charfl)]
    set line [format "%s" $asmpar(charfl)]
    set line_length_total [string length $line]
    set comment "\\characterization data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### ncat: number of thresholds or categories
    set comment "\\number of thresholds or categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(ncat) $comment]

    ### cat(i): threshold values or category codes
    set line_length_total 0
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $asmpar(cat,$ii)] "
	set line [format "%g" $asmpar(cat,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\thresholds or categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icoli(i): column number for the characterization variables
    set line_length_total 0
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%d" [$char_vals($ii,combo) curselection]] "
	set line [format "%d" [$char_vals($ii,combo) curselection]]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set comment "\\column number for the characterization variables"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### mean: mean value to consider (0=no, 1=yes)
    set comment "\\mean value to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(mean) $comment]

    ### dist: value from the distribution to consider (0=no, 1=yes)
    set comment "\\value from the distribution to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(dist) $comment]

    ### cmean: column number for the mean
    set comment "\\column number for the mean"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$mean_value_combo_w curselection] $comment]

    ### cdist: column number for the standard deviation
    set comment "\\column number for the standard deviation"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$distribution_value_combo_w curselection] $comment]

    ### nsimchar: number of characterization realizations to consider
    set comment "\\number of characterization realizations to consider"
    catch {scan [$realization_char_w get] {%d} asmpar(nsimchar)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(nsimchar) $comment]

    ### outfl: file for output data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(outfl)]
    set line [format "%s" $asmpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### noutfl: number of output files
    set comment "\\number of output files"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(noutfl) $comment]

    ### sname: extensions of output files
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(sname)]
    set line [format "%s" $asmpar(sname)]
    set line_length_total [string length $line]
    set comment "\\extensions of output files"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### reset the cursor on arrow 
    . configure -cursor arrow
    update

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc write_export_asm_params_run { fname } {
    global tcl_platform
    global version
    global asmpar
    global export_asm_box_w
    global variable_type_value_w
    global char_vals
    global mean_value_combo_w
    global distribution_value_combo_w

    ### message, if file cannot be open
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    log "EXPORT ASM, PMWIN Version: $version\n"
    log "Parameters:\n"
    
    puts $outf "Parameters for EXPORT_ASM_PMWIN"
    puts $outf "*******************************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with simulated realizations
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(datafl)]
    set line [format "%s" $asmpar(datafl)]
    set line_length_total [string length $line]
    set comment "\\simulated realizations"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file with simulated realizations = $asmpar(datafl)"

    ### nsim: number of realizations to consider
    set comment "\\number of realizations to consider"
    catch {scan [$realization_sim_w get] {%d} asmpar(nsim)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(nsim) $comment]
    log "number of realizations to consider = $asmpar(nsim)"

    ### vartype: variable type (1=cont.(cdf), 0=cat.(pdf)
    set comment "\\variable type (1=cont.(cdf), 0=cat.(pdf)"
    set line [format "%s" $asmpar(vartype)] 
    if { $line == "Categorical" } {
	set asmpar(vartype) 0
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(vartype) $comment]
	set asmpar(vartype) "Categorical"
    } elseif { $line == "Continous" } {
	set asmpar(vartype) 1
	puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(vartype) $comment]
	set asmpar(vartype) "Continous"
    }
    log "variable type (1=continous, 0=categorical) = $asmpar(vartype)"

    ### nx, ny, nz: number of grid cells (x-, y-, and z-axis)
    puts -nonewline $outf [format "%d %d %d" $asmpar(nx) $asmpar(ny) $asmpar(nz)]
    set line [format "%d %d %d" $asmpar(nx) $asmpar(ny) $asmpar(nz)]
    set line_length [string length $line]
    set comment "\\number of grid cells (x-, y-, and z-axis)"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]
    log "number of grid cells (x-, y-, z-axis) = $asmpar(nx)  $asmpar(ny)  $asmpar(nz)"


    ### charfl: file with characterization data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(charfl)]
    set line [format "%s" $asmpar(charfl)]
    set line_length_total [string length $line]
    set comment "\\characterization data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr ($parameter_space - 0) - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file with characterization data = $asmpar(charfl)"

    ### ncat: number of thresholds or categories
    set comment "\\number of thresholds or categories"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(ncat) $comment]
    log "number of categories = $asmpar(ncat)"

    ### cat(i): threshold values or category codes
    set line_length_total 0
    log_nonewline "number of categories = "
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%g" $asmpar(cat,$ii)] "
	set line [format "%g" $asmpar(cat,$ii)]
	log_nonewline "$asmpar(cat,$ii)  "
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    log_newline
    set comment "\\thresholds or categories"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    
    ### icoli(i): column number for the characterization variables
    set line_length_total 0
    log_nonewline "column number of the characterization variables = "
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	puts -nonewline $outf "[format "%d" [$char_vals($ii,combo) curselection]] "
	set line [format "%d" [$char_vals($ii,combo) curselection]]
	log_nonewline "[$char_vals($ii,combo) curselection]  "
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    log_newline
    set comment "\\column number for the characterization variables"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### mean: mean value to consider (0=no, 1=yes)
    set comment "\\mean value to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(mean) $comment]
    log "mean value to consider (0=no, 1=yes) = $asmpar(mean)"

    ### dist: value from the distribution to consider (0=no, 1=yes)
    set comment "\\value from the distribution to consider (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(dist) $comment]
    log "value from the distribution to consider (0=no, 1=yes) = $asmpar(dist)"

    ### cmean: column number for the mean
    set comment "\\column number for the mean"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$mean_value_combo_w curselection] $comment]
    log "column number for the mean = [$mean_value_combo_w curselection]"

    ### cdist: column number for the standard deviation
    set comment "\\column number for the standard deviation"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" [$distribution_value_combo_w curselection] $comment]
    log "column number for the standard deviation = [$distribution_value_combo_w curselection]"

    ### nsimchar: number of characterization realizations to consider
    set comment "\\number of characterization realizations to consider"
    catch {scan [$realization_char_w get] {%d} asmpar(nsimchar)}
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(nsimchar) $comment]
    log "number of characterization realizations to consider = $asmpar(nsimchar)"

    ### outfl: file/s for output data
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(outfl)]
    set line [format "%s" $asmpar(outfl)]
    set line_length_total [string length $line]
    set comment "\\output data"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "file/s for output data = $asmpar(outfl)"

    ### noutfl: number of output file/s
    set comment "\\number of output file/s"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $asmpar(noutfl) $comment]
    log "number of output file/s = $asmpar(noutfl)"

    ### sname: extensions of output file/s
    puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $asmpar(sname)]
    set line [format "%s" $asmpar(sname)]
    set line_length_total [string length $line]
    set comment "\\extensions of output files"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }
    log "extensions of output file/s = $asmpar(sname)"

    ### close the file
    close $outf
}
#============================================================================


#############################################################################
proc export_asm_continous_variables_2d { iv } {
    global asmpar 
    global outf
    global inf

    ### searches the line "simulated value" and reject all preceded lines
    if { $iv == 1 } {
	set line [gets $inf]
	while { $line != "Simulated Value" } {
	    set line [gets $inf]
	}
	
	### set the grid indices and the matrix for the first realization
	if { $asmpar(nx) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(nx) $asmpar(ny)]
	    for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	}
    } elseif { $iv > 1 } {
	### set the grid indices and the matrix for the following realizations
	if { $asmpar(nx) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outf [format "%10d %10d" $asmpar(nx) $asmpar(ny)]
	    for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $inf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%10g " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%10g" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%10g" $line]
			set iii 0
		    }
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_asm_continous_variables_3d { iv v } {
    global asmpar 
    global outfcon3d
    global inf

    ### searches the line "simulated value" and reject all preceded lines
    if { $iv == 1 && $v == 1 } {
	set line [gets $inf]
	while { $line != "Simulated Value" } {
	    set line [gets $inf]
	}
	
	### set the grid indices and the matrix for the first realization and layer
	puts $outfcon3d($iv,$v) [format "%10d %10d" $asmpar(nx) $asmpar(ny)]
	for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $inf]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcon3d($iv,$v) [format "%10g " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcon3d($iv,$v) [format "%10g" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcon3d($iv,$v) [format "%10g" $line]
		    set iii 0
		}
	    }
	}
    } elseif { ($iv == 1 && $v > 1) || $iv > 1 } {
	### set the grid indices and the matrix for the following realizations and layers
	puts $outfcon3d($iv,$v) [format "%10d %10d" $asmpar(nx) $asmpar(ny)]
	for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $inf]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcon3d($iv,$v) [format "%10g " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcon3d($iv,$v) [format "%10g" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcon3d($iv,$v) [format "%10g" $line]
		    set iii 0
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc mean_value_replacement { } {
    global asmpar 
    global outf
    global inf
    global charf
    global repf
    global mean_value_combo_w
    global char_vals
    global mean_position
    global variable_position
    global mean_value
    global ln_mean_value

    ### searches the mean value position in the characterization data
    set mean_position [$mean_value_combo_w curselection]

    ### reading the mean values of the variables
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	set variable_position($ii) [$char_vals($ii,combo) curselection]
	for { set i 0 } { $i <= [expr + $asmpar(charvar) + $variable_position($ii) + 3] } { incr i } {
	    set charline [gets $charf]
	}
	for { set iii 0 } { $iii <= $mean_position } { incr iii } {
	    set space [string first " " $charline]
	    set ln_mean_value($ii) [format "%g" [string range $charline 0 [expr $space - 1]]] 
	    ;# it is the ln-mean of the field data
	    set charline [string range $charline [expr $space + 1] end]
	}
	set mean_value($ii) [expr (0 + exp($ln_mean_value($ii)))] ;# the ln-value has to be expound to the base of e
	seek $charf 0
    }

    ### searches the line "simulated value" and reject all preceded lines
    set line [gets $inf]
    puts $repf $line
    while { $line != "Simulated Value" } {
	set line [gets $inf]
	puts $repf $line
    }
    
    ### replace the categorical variables with mean values
    while { ![eof $inf] } {
	set line [gets $inf]
	catch {scan $line {%d} line}
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    if { $line == $asmpar(cat,$ii) } {
		puts $repf $mean_value($ii)
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_asm_categorical_variables_mean_2d { i } {
    global asmpar 
    global outf
    global charf
    global repf

    ### searches the line "simulated value" and reject all preceded lines
    if { $i == 1 } {
	seek $repf 0
	set line [gets $repf]
	while { $line != "Simulated Value" } {
	    set line [gets $repf]
	}
	
	### set the grid indices and the matrix for the first realization
	if { $asmpar(nx) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	    for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	}
    } elseif { $i > 1 } {
	### set the grid indices and the matrix for the following realizations
	if { $asmpar(nx) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outf [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	    for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outf [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outf [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outf [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_asm_categorical_variables_mean_3d { iv v } {
    global asmpar 
    global outfcatm3d
    global charf
    global repf

    ### searches the line "simulated value" and reject all preceded lines
    if { $iv == 1 && $v == 1 } {
	seek $repf 0
	set line [gets $repf]
	while { $line != "Simulated Value" } {
	    set line [gets $repf]
	}
	
	### set the grid indices and the matrix for the first realization
	puts $outfcatm3d($iv,$v) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $repf]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcatm3d($iv,$v) [format "%12.4e " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcatm3d($iv,$v) [format "%12.4e" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcatm3d($iv,$v) [format "%12.4e" $line]
		    set iii 0
		}
	    }
	}
    } elseif { ($iv == 1 && $v > 1) || $iv > 1 } {
	### set the grid indices and the matrix for the following realizations
	puts $outfcatm3d($iv,$v) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	for { set i 1 } { $i <= $asmpar(ny) } { incr i } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $repf]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcatm3d($iv,$v) [format "%12.4e " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcatm3d($iv,$v) [format "%12.4e" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcatm3d($iv,$v) [format "%12.4e" $line]
		    set iii 0
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc distribution_value_replacement { i } {
    global asmpar 
    global outf
    global inf
    global charf
    global repfd
    global mean_value_combo_w
    global distribution_value_combo_w
    global char_vals
    global mean_position
    global std_position
    global variable_position
    global mean_value
    global ln_mean_value
    global std_value
    global ln_std_value

    ### go to the start of the simulated realizations file
    seek $inf 0

    ### searches the mean value position in the characterization data file
    set mean_position [$mean_value_combo_w curselection]

    ### searches the standard deviation position in the characterization data file
    set std_position [$distribution_value_combo_w curselection]

    ### reading the mean values and the standard deviations of the variables
    for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	set variable_position($ii) [$char_vals($ii,combo) curselection]
	for { set v 0 } { $v <= [expr + $asmpar(charvar) + $variable_position($ii) + 3] } { incr v } {
	    set charline_1 [gets $charf]
	    set charline_2 $charline_1
	}
	for { set iii 0 } { $iii <= $mean_position } { incr iii } {
	    set space [string first " " $charline_1]
	    set mean_value($ii) [format "%g" [string range $charline_1 0 [expr $space - 1]]]
	    set charline_1 [string range $charline_1 [expr $space + 1] end] ;# it is the ln-mean of the field data
	}
	for { set iv 0 } { $iv <= $std_position } { incr iv } {
	    set space [string first " " $charline_2]
	    set std_value($ii) [format "%g" [string range $charline_2 0 [expr $space - 1]]] ;# it is the std of the ln-mean
	    set charline_2 [string range $charline_2 [expr $space + 1] end]
	}
	seek $charf 0
    }

    ### searches the line "simulated value" and reject all preceded lines
    set line [gets $inf]
    puts $repfd($i) $line
    while { $line != "Simulated Value" } {
	set line [gets $inf]
	puts $repfd($i) $line
    }
    
    ### replace the categorical variables with normally distributed values values 
    while { ![eof $inf] } {
	set line [gets $inf]
	catch {scan $line {%d} line}
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    if { $line == $asmpar(cat,$ii) } {
		puts $repfd($i) [expr (0 + exp([geossav_grandom -sigma $std_value($ii) -mean $mean_value($ii)]))]
		;# the value given back from the geossav_grandom proc (g for gauss) in the file geossav.c 
		;# is a ln-value from a normal distribution, so this value has to be expound to the base of e
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_asm_categorical_variables_distribution_2d { i vi } {
    global asmpar 
    global outfcatd2d
    global charf
    global repf

    ### searches the line "simulated value" and reject all preceded lines
    if { $i == 1 } {
	seek $repf($vi) 0
	set line [gets $repf($vi)]
	while { $line != "Simulated Value" } {
	    set line [gets $repf($vi)]
	}
	
	### set the grid indices and the matrix for the first realization
	if { $asmpar(nx) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	    for { set iv 1 } { $iv <= $asmpar(ny) } { incr iv } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	}
    } elseif { $i > 1 } {
	### set the grid indices and the matrix for the following realizations
	if { $asmpar(nx) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(ny) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(ny) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(ny) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(ny) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(ny) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(nz)]
	    for { set i 1 } { $i <= $asmpar(nz) } { incr i } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	} elseif { $asmpar(nz) == 1 } {
	    puts $outfcatd2d($i,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	    for { set iv 1 } { $iv <= $asmpar(ny) } { incr iv } {
		set iii 0
		for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		    set line [gets $repf($vi)]
		    catch {scan $line {%g} line}
		    set iii [expr $iii + 1]
		    if { $iii <= 19 } {
			if { $ii < $asmpar(nx) } {
			    puts -nonewline $outfcatd2d($i,$vi) [format "%12.4e " $line]
			} elseif { $ii == $asmpar(nx) } {
			    puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			}
		    } elseif { $iii == 20 } {
			puts $outfcatd2d($i,$vi) [format "%12.4e" $line]
			set iii 0
		    }
		}
	    }
	}
    }
}
#============================================================================


#############################################################################
proc export_asm_categorical_variables_distribution_3d { i v vi } {
    global asmpar
    global outfcatd3d
    global charf
    global repfd

    ### searches the line "simulated value" and reject all preceded lines
    if { $i == 1 && $v == 1 } {
	seek $repfd($vi) 0
	set line [gets $repfd($vi)]
	while { $line != "Simulated Value" } {
	    set line [gets $repfd($vi)]
	}
	
	### set the grid indices and the matrix for the first realization
	puts $outfcatd3d($i,$v,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	for { set iv 1 } { $iv <= $asmpar(ny) } { incr iv } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $repfd($vi)]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcatd3d($i,$v,$vi) [format "%12.4e " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcatd3d($i,$v,$vi) [format "%12.4e" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcatd3d($i,$v,$vi) [format "%12.4e" $line]
		    set iii 0
		}
	    }
	}
    } elseif { ($i == 1 && $v > 1) || $i > 1 } {
	### set the grid indices and the matrix for the following realizations
	puts $outfcatd3d($i,$v,$vi) [format "%12d %12d" $asmpar(nx) $asmpar(ny)]
	for { set iv 1 } { $iv <= $asmpar(ny) } { incr iv } {
	    set iii 0
	    for { set ii 1 } { $ii <= $asmpar(nx) } { incr ii } {
		set line [gets $repfd($vi)]
		catch {scan $line {%g} line}
		set iii [expr $iii + 1]
		if { $iii <= 19 } {
		    if { $ii < $asmpar(nx) } {
			puts -nonewline $outfcatd3d($i,$v,$vi) [format "%12.4e " $line]
		    } elseif { $ii == $asmpar(nx) } {
			puts $outfcatd3d($i,$v,$vi) [format "%12.4e" $line]
		    }
		} elseif { $iii == 20 } {
		    puts $outfcatd3d($i,$v,$vi) [format "%12.4e" $line]
		    set iii 0
		}
	    }
	}
    }
}
#============================================================================

#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_export_asm_main { } {
    global exedir
    source $exedir/Geossav_export_asm_pmwin_globals.tcl
    
    . configure -cursor watch
    update
    
    set export_asm_box_w [toplevel .export_asm_box -class Dialog]
    wm resizable $export_asm_box_w false false
    wm sizefrom $export_asm_box_w program
    wm group $export_asm_box_w .
    wm transient $export_asm_box_w [winfo toplevel [winfo parent $export_asm_box_w]]
    wm title $export_asm_box_w "EXPORT ASM, PMWIN parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $export_asm_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set export_asm_tabnotebook_w [iwidgets::tabnotebook $export_asm_box_w.export_asm_tbn \
				      -width 5.4i \
				      -height 4.0i \
				      -tabpos n \
				      -tabforeground gray50 \
				      -background gray85 \
				      -bevelamount 2]
    
    pack $export_asm_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav EXPORT_ASM-parameters: Files (there is only one page)
    #
    set page [$export_asm_box_w.export_asm_tbn add \
		  -label "Files"]
    

    # initialize dimension names for characterization data
    if { ![export_asm_read_char_data $asmpar(charfl)] } {
	set dim_name_char {}
	for { set i 1} { $i <= 3 } { incr i } {
	    lappend dim_name_char $i
	}
    }
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    # fields for simulated data
    set data_iframe_w [iwidgets::labeledframe $work_frame_w.data \
			   -labeltext "Simulation Input" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set data_frame_w [$data_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $data_frame_w.subfr1]

    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "Simulated realizations, file " \
		      -width 51 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable asmpar(datafl)]

    if { $asmpar(datafl) != {*.out} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }

    if { $asmpar(datafl) == {*.out} } {
	set asmpar(dflbc) 0
    } elseif { $asmpar(datafl) != {*.out} } {
	set asmpar(dflbc) $asmpar(dflbc)
    }
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".out" \
						    -filetypes {{{Simulation Files} {.out}} {{All Files} {.*}}} \
						    -initialfile $asmpar(datafl) \
						    -initialdir $exedir \
						    -title "Select simulated realizations file for EXPORT ASM, PMWIN"]
				 
				 ### counts the activation of the datafl browse button
				 set asmpar(dflbc) [expr $asmpar(dflbc) + 1]
				 
 				 ### select file name
				 if { $tmp_fname != {} } {
				     set asmpar(datafl) $tmp_fname

				     ### set datafl and realization widgets normal
				     if { $asmpar(datafl) == {*.out} } {
					 $datafl_w configure \
					     -state disabled -textbackground gray85 -foreground gray50
					 $realization_sim_w configure \
					     -state disabled -textbackground gray85 -foreground gray50
					 $charfl_browse_w configure -state disabled
				     } elseif { $asmpar(datafl) != {*.out} } {
					 $datafl_w configure -state normal -textbackground white -foreground black
					 $realization_sim_w configure -state normal -textbackground white -foreground black
					 $charfl_browse_w configure -state normal
				     }
				     
				     ### create infofl path- and filename
				     set infofl_rootname [file rootname $asmpar(datafl)]
				     set infofl_dirname [file dirname $asmpar(datafl)]
				     set infofl_dirname_lenght [string length $infofl_dirname]
				     set infofl_name [string range $infofl_rootname \
							  [expr $infofl_dirname_lenght + 1] end]
				     set infofl_extension [file extension $asmpar(datafl)]
				     set info "/Info_"
				     regsub {} $infofl_extension $infofl_name infofl_1
				     regsub {} $infofl_1 $info infofl_2
				     regsub {} $infofl_2 $infofl_dirname infofl_3
				     set asmpar(infofl) $infofl_3
				     
				     ### open infofl and read data
				     if { $asmpar(infofl) != {} } {
					 read_infofl_params $asmpar(infofl)
				     }
				     
				     ### output path- and filename, and number of files
				     if { $asmpar(datafl) != {*.dat} } {
					 set datafl_rootname [file rootname $asmpar(datafl)]
					 set datafl_dirname [file dirname $asmpar(datafl)]
					 set datafl_dirname_lenght [string length $datafl_dirname]
					 set datafl_name \
					     [string range $datafl_rootname [expr $datafl_dirname_lenght + 1] end]
					 
					 set outfl_extension [file extension $asmpar(outfl)]
					 set export "/EXPORT_"
					 regsub {} $outfl_extension $datafl_name outputfl_1
					 regsub {} $outputfl_1 $export outputfl_2
					 regsub {} $outputfl_2 $datafl_dirname outputfl_3
					 set asmpar(outfl) $outputfl_3
				     }
				     number_output_files
				 }
			     }]
			 
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0


    set subframe2_w [frame $data_frame_w.subfr2]

    set realization_sim_w [iwidgets::entryfield $subframe2_w.realization_sim \
			       -labeltext "Number of realizations " \
			       -width 8 \
			       -state disabled \
			       -textbackground gray85 \
			       -foreground gray50 \
			       -textvariable asmpar(nsim) \
			       -validate {check_number_nsim_sim %W %P %c}]
    if { $asmpar(datafl) != {*.out} } {
	$realization_sim_w configure -state normal -textbackground white -foreground black
    }


    set variable_type_label_w [iwidgets::labeledwidget $subframe2_w.label_variable_type \
				   -labeltext "Variable type  "]
    set tmp_vartype $asmpar(vartype)
    set variable_type_value_w [iwidgets::labeledwidget $subframe2_w.value_variable_type \
				   -labelvariable asmpar(vartype)]
    if { $tmp_vartype == 0 } {
	set asmpar(vartype) "Categorical"
    } elseif { $tmp_vartype == 1 } {
    	set asmpar(vartype) "Continous"
    }


    grid columnconfigure $subframe2_w 0 -minsize 186
    grid columnconfigure $subframe2_w 1 -minsize 140
    grid columnconfigure $subframe2_w 2 -minsize 60
    grid $realization_sim_w     -row 0 -column 0 -sticky e -pady 0
    grid $variable_type_label_w -row 0 -column 1 -sticky e -pady 0
    grid $variable_type_value_w -row 0 -column 2 -sticky w -pady 0
    bind [$realization_sim_w component entry]  <KeyRelease> {number_output_files}


    set subframe3_w [frame $data_frame_w.subfr3]

    set number_x_label_w [iwidgets::labeledwidget $subframe3_w.label_number_x \
				   -labeltext "Number x"]
    set number_x_value_w [iwidgets::labeledwidget $subframe3_w.value_number_x \
				   -labelvariable asmpar(nx)]


    set number_y_label_w [iwidgets::labeledwidget $subframe3_w.label_number_y \
				   -labeltext "Number y"]
    set number_y_value_w [iwidgets::labeledwidget $subframe3_w.value_number_y \
				   -labelvariable asmpar(ny)]


    set number_z_label_w [iwidgets::labeledwidget $subframe3_w.label_number_z \
				   -labeltext "Number z"]
    set number_z_value_w [iwidgets::labeledwidget $subframe3_w.value_number_z \
				   -labelvariable asmpar(nz)]


    grid columnconfigure $subframe3_w 0 -minsize 129
    grid columnconfigure $subframe3_w 1 -minsize 35
    grid columnconfigure $subframe3_w 2 -minsize 100
    grid columnconfigure $subframe3_w 3 -minsize 40
    grid columnconfigure $subframe3_w 4 -minsize 100
    grid columnconfigure $subframe3_w 5 -minsize 35
    grid $number_x_label_w -row 0 -column 0 -sticky e -pady 0
    grid $number_x_value_w -row 0 -column 1 -sticky w -pady 0
    grid $number_y_label_w -row 0 -column 2 -sticky e -pady 0
    grid $number_y_value_w -row 0 -column 3 -sticky w -pady 0
    grid $number_z_label_w -row 0 -column 4 -sticky e -pady 0
    grid $number_z_value_w -row 0 -column 5 -sticky w -pady 0


    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for characterization data
    set characterization_iframe_w [iwidgets::labeledframe $work_frame_w.characterization \
				       -labeltext "Characterization Input" \
				       -labelpos nw \
				       -borderwidth 2 \
				       -relief groove]
    set characterization_frame_w [$characterization_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $characterization_frame_w.subfr1]

    set charfl_w [iwidgets::entryfield $subframe1_w.charfl \
		      -labeltext "Variables characterization, file  " \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -width 47 \
		      -textvariable asmpar(charfl)]


    set charfl_browse_w [iwidgets::pushbutton $subframe1_w.charfl_browse \
			     -state disabled \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".gtd" \
						    -filetypes {{{Geo-EAS Files} {.gtd}} {{All Files} {.*}}} \
						    -initialfile $asmpar(charfl) \
						    -initialdir $exedir \
						    -title "Select ground type characterization file for EXPORT ASM, PMWIN"]

				 ### exit from the BICALIB parameters tabnotebook
				 set asmpar(cancel_counter_characterization) 0
				 
				 ### set the old number of characterization variables
				 if { $asmpar(charfl) != {*.gtd} } {
				     set asmpar(oldcharvar) $asmpar(charvar)
				 }

				 ### select file name
				 if { $tmp_fname != {} } {
				     set asmpar(charfl) $tmp_fname
				     unset dim_name_char
				     if { [export_asm_read_char_data $asmpar(charfl)] != 1 } {
					 set dim_name_char {}
					 for { set i 1 } { $i <= $asmpar(charvar) } { incr i } {
					     lappend dim_name_char $i
					 }
				     }
				 }

				 ### fills part of the char table
				 for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
				     $char_vals($ii,entry) configure -state normal -textbackground white -foreground black
				     $char_vals($ii,combo) configure -state normal -textbackground white -foreground black
				     $char_vals($ii,combo) clear list
				     foreach i $dim_name_char {
					 $char_vals($ii,combo) insert list end $i
				     }
				     $char_vals($ii,combo) selection set [expr $ii - 1]
				     set asmpar(icoli) [lindex $dim_name_char $ii]
				     update
				     $char_table_w window configure $ii,2 -window $char_vals($ii,combo)
				 }
				 
				 ### fills the mean_value_combobox
				 $mean_value_combo_w configure -state normal -textbackground white -foreground black
				 $mean_value_combo_w clear list
				 foreach i $dim_name_char {
				     $mean_value_combo_w insert list end $i
				 }
				 $mean_value_combo_w selection set [expr $asmpar(ncat) + 0]
				 set asmpar(cmean) [lindex $dim_name_char [expr $asmpar(ncat) + 0]]
				 update

				 ### fills the distribution_value_combobox
				 $distribution_value_combo_w configure \
				     -state normal -textbackground white -foreground black
				 $distribution_value_combo_w clear list
				 foreach i $dim_name_char {
				     $distribution_value_combo_w insert list end $i
				 }
				 $distribution_value_combo_w selection set [expr $asmpar(ncat) + 1]
				 set asmpar(cdist) [lindex $dim_name_char [expr $asmpar(ncat) + 1]]
				 update
				 if { $asmpar(mean) == 1 } {
				     $distribution_value_combo_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 } elseif { $asmpar(mean) == 0 } {
				     set asmpar(dist) 1
				     $distribution_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 }

				 ### set the rest of the characterization input widgets normal/disabled
				 $charfl_w             configure -state normal -textbackground white -foreground black
				 $charfl_browse_w      configure -state normal
				 $mean_value_w         configure -state normal
				 $distribution_value_w configure -state normal
				 if { $asmpar(mean) == 1 } {
				     $mean_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $distribution_value_combo_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				     $realization_char_w configure \
					 -state disabled -textbackground gray85 -foreground gray50
				 } elseif { $asmpar(dist) == 1 } {
				     $mean_value_combo_w configure \
					 -state normal -textbackground white -foreground black
				     $distribution_value_combo_w \
					 configure -state normal -textbackground white -foreground black
				     $realization_char_w \
					 configure -state normal -textbackground white -foreground black
				 }

				 ### number of output files
				 number_output_files

				 ### message, if the variables characterization file has to few data
				 if { $asmpar(datafl) != {*.out} && $asmpar(charfl) != {*.gtd} } {
				     if { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
					 tk_messageBox \
					     -title "Characterization Input" \
					     -icon warning \
					     -message "The variables characterization file has to few data.\nYou should select another file." \
					     -type ok
				     }
				 }

			     }]

    ### set charfl disabled/normal
    if { $asmpar(datafl) != {*.out} } {
	$charfl_browse_w configure -state normal
    }    

    ### set charfl disabled/normal and count the exit from the tabnotebook
    if { $asmpar(charfl) != {*.gtd} } {
	$charfl_w configure -state normal -textbackground white -foreground black
	set asmpar(cancel_counter_characterization) [expr $asmpar(cancel_counter_characterization) + 1]
    }

    ### set charfl disabled/normal depending on the vartype
    if { $asmpar(vartype) == "Continous" } {
	$charfl_w configure -state disabled -textbackground gray85 -foreground gray50
	$charfl_browse_w configure -state disabled
    }


    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $charfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $charfl_browse_w -row 0 -column 1 -sticky w -pady 0


    set subframe2_w [frame $characterization_frame_w.subfr2]

    ### setting the characterization table normal/disabled depending on the input data
    set char_frame_w [frame $subframe2_w.char_frame \
			  -background grey40 \
			  -borderwidth 2 \
			  -relief sunken]
    
    set char_sframe_w [frame $char_frame_w.char_sframe \
			   -height 72 \
			   -width 310]
    set char_hsbar_w [scrollbar $char_sframe_w.yscroll \
			  -orient vertical \
			  -command "$char_sframe_w.table yview"]
    
    
    if { $asmpar(datafl) == "*.out" && $asmpar(charfl) == "*.gtd" } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 3 \
			      -rows 4 \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 19
	$char_table_w width 2 29
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	
	for {set ii 1} {$ii <= 3} {incr ii} {
	    
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set asmpar(cat,$ii) $ii
	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable asmpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state disabled \
					  -width 19 \
					  -textbackground gray85 \
					  -foreground gray50]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable asmpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 29 \
					  -listheight 100]
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
	    }
	    $char_vals($ii,combo) selection set [expr $ii - 1]
	    set asmpar(icoli) [lindex $dim_name_char $ii]
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    
	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $asmpar(vartype) == "Continous" } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    } elseif { ($asmpar(datafl) != "*.out") && ($asmpar(charfl) != "*.gtd") || \
	     ($asmpar(datafl) == "*.out") && ($asmpar(charfl) != "*.gtd") } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 3 \
			      -rows [expr $asmpar(ncat) + 1] \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 19
	$char_table_w width 2 29
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	    
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
		
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable asmpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state normal \
					  -width 19 \
					  -textbackground white \
					  -foreground black]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable asmpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 29 \
					  -listheight 100]
	    set tmp_icoli($ii) $asmpar(icoli,$ii)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if asmpar(icoli) is empty
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
		$char_vals($ii,combo) delete list [expr $asmpar(charvar) + 1]
		if { $i == $tmp_icoli($ii) } { ;# ok, that's the value in asmpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    set sel [expr $sel - ($asmpar(cancel_counter_characterization) * ($asmpar(charvar) + 1))]
	    $char_vals($ii,combo) selection set $sel
	    set asmpar(icoli,$ii) [lindex $dim_name_char $sel]
	    $char_vals($ii,combo) configure -state normal -textbackground white -foreground black
	    
	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $asmpar(vartype) == "Continous" } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    } elseif { $asmpar(datafl) != "*.out" && $asmpar(charfl) == "*.gtd" } {
	
	set char_table_w [table $char_sframe_w.table \
			      -variable char_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 3 \
			      -rows [expr $asmpar(ncat) + 1] \
			      -yscrollcommand "$char_hsbar_w set" \
			      -height 4]
	
	$char_table_w width 0 5
	$char_table_w width 1 19
	$char_table_w width 2 29
	
	set char_vals(0,0) "\#"
	set char_vals(0,1) "Category"
	set char_vals(0,2) "Variable"
	
	for { set ii 1 } { $ii <= $asmpar(ncat) } { incr ii } {
	    
	    set char_vals($ii,label) [iwidgets::labeledwidget $char_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]

	    set char_vals($ii,entry) [iwidgets::entryfield $char_table_w.entry$ii \
					  -textvariable asmpar(cat,$ii) \
					  -validate {regexpvalidate %c} \
					  -state disabled \
					  -width 19 \
					  -textbackground gray85 \
					  -foreground gray50]
	    
	    set char_vals($ii,combo) [iwidgets::combobox $char_table_w.combo$ii \
					  -textvariable asmpar(icoli,$ii) \
					  -validate {regexpvalidate %c} \
					  -width 29 \
					  -listheight 100]
	    foreach i $dim_name_char {
		$char_vals($ii,combo) insert list end $i
	    }
	    $char_vals($ii,combo) selection set [expr $ii - 1]
	    set asmpar(icoli) [lindex $dim_name_char $ii]
	    $char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    
	    $char_table_w window configure $ii,0 -window $char_vals($ii,label) -relief flat -background gray50
	    $char_table_w window configure $ii,1 -window $char_vals($ii,entry)
	    $char_table_w window configure $ii,2 -window $char_vals($ii,combo)

	    ### set entryfield and combobox of the characterizaton table disabled/normal depending on the vartype
	    if { $asmpar(vartype) == "Continous" } {
		$char_vals($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
		$char_vals($ii,combo) configure -state disabled -textbackground gray85 -foreground gray50
	    }
	}
    }


    ### set the mean value checkbutton
    set mean_value_w [checkbutton $subframe2_w.mean_cbutton \
			  -text "Mean value ( ln ! )      " \
			  -variable asmpar(mean) \
			  -state disabled \
			  -command {mean_value_selection}]
    if { $asmpar(charfl) != {*.gtd} } {
	$mean_value_w configure -state normal
    }


    ### set the distribution value checkbutton
    set distribution_value_w [checkbutton $subframe2_w.distribution_cbutton \
				  -text "Value from distribution" \
				  -variable asmpar(dist) \
				  -state disabled \
				  -command {distribution_value_selection}]
    if { $asmpar(charfl) != {*.gtd} } {
	$distribution_value_w configure -state normal
    }

    ### set the mean value combobox
    set mean_value_combo_w [iwidgets::combobox $subframe2_w.mean_combo \
				-labeltext "Mean" \
				-textvariable asmpar(cmean) \
				-width 17 \
				-listheight 80 \
				-validate {regexpvalidate %c}]
    if { $asmpar(charfl) == {*.gtd} } {
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	}
	$mean_value_combo_w selection set 0
	set asmpar(cmean) [lindex $dim_name_char 0]
	$mean_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $asmpar(charfl) != {*.gtd} } {
	$mean_value_combo_w configure -state normal -textbackground white -foreground black
	set tmp_cmean $asmpar(cmean)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	foreach i $dim_name_char {
	    $mean_value_combo_w insert list end $i
	    $mean_value_combo_w delete list [expr $asmpar(charvar) + 1]
	    if { $i == $tmp_cmean } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($asmpar(cancel_counter_characterization) * ($asmpar(charvar) + 1))]
	$mean_value_combo_w selection set $sel
	set asmpar(cmean) [lindex $dim_name_char $sel]
	update
    }
    
    ### set mean- and distribution checkbutton, and mean value combobox disabled/normal depending on the vartype
    if { $asmpar(vartype) == "Continous" } {
	$mean_value_w configure -state disabled
	$distribution_value_w configure -state disabled
	$mean_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    }

    pack $char_hsbar_w  -side right -anchor w -fill y
    pack $char_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $char_sframe_w -anchor nw -expand 0
    pack propagate $char_sframe_w 0

    grid $char_frame_w -row 0 -column 0 -sticky new -rowspan 4 -pady 0
    grid columnconfigure $subframe2_w 1 -minsize 50
    grid columnconfigure $subframe2_w 2 -minsize 50
    grid columnconfigure $subframe2_w 3 -minsize 120

    grid $mean_value_w         -row 0 -column 1 -sticky e -pady 0 -padx 2
    grid $distribution_value_w -row 1 -column 1 -sticky e -pady 0 -padx 2
    grid $mean_value_combo_w   -row 2 -column 1 -sticky e -pady 3 -padx 5

    
    set subframe3_w [frame $characterization_frame_w.subfr3]

    ### set the realization widget
    set realization_char_w [iwidgets::entryfield $subframe3_w.realization_char \
				-labeltext "   Number of characterizations  " \
				-width 8 \
				-state disabled \
				-textbackground gray85 \
				-foreground gray50 \
				-textvariable asmpar(nsimchar) \
				-validate {check_number_nsim_char %W %P %c}]
    if { $asmpar(charfl) != {*.gtd} } {
	$realization_char_w configure -state normal -textbackground white -foreground black
    }

    ### set the distribution value combobox
    set distribution_value_combo_w [iwidgets::combobox $subframe3_w.distribution_combo \
					-labeltext "Standard Deviation" \
					-textvariable asmpar(cdist) \
					-width 17 \
					-listheight 80 \
					-validate {regexpvalidate %c}]
    if { $asmpar(charfl) == {*.gtd} } {
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	}
	$distribution_value_combo_w selection set 1
	set asmpar(cdist) [lindex $dim_name_char 1]
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $asmpar(charfl) != {*.gtd} } {
	$distribution_value_combo_w configure -state normal -textbackground white -foreground black
	set tmp_cdist $asmpar(cdist)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if sisimpar(icolsz) is empty
	foreach i $dim_name_char {
	    $distribution_value_combo_w insert list end $i
	    $distribution_value_combo_w delete list [expr $asmpar(charvar) + 1]
	    if { $i == $tmp_cdist } { ;# ok, that's the value in sisimpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($asmpar(cancel_counter_characterization) * ($asmpar(charvar) + 1))]
	$distribution_value_combo_w selection set $sel
	set asmpar(cdist) [lindex $dim_name_char $sel]
	update
    }


    ### set the distribution_value_combobox disabled/normal depending on the value selection
    if { $asmpar(charfl) != {*.gtd} } {
	if { $asmpar(mean) == 1 } {
	    set asmpar(dist) 0
	    $distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
	    $realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
	} elseif { $asmpar(mean) == 0 } {
	    set asmpar(dist) 1
	    $distribution_value_combo_w configure -state normal -textbackground white -foreground black
	    $realization_char_w configure -state normal -textbackground white -foreground black
	}
    }

    ### set mean- and distribution checkbutton, and mean value combobox disabled/normal depending on the vartype
    if { $asmpar(vartype) == "Continous" } {
	$realization_char_w configure -state disabled -textbackground gray85 -foreground gray50
	$distribution_value_combo_w configure -state disabled -textbackground gray85 -foreground gray50
    }


    grid columnconfigure $subframe3_w 0 -minsize 250
    grid columnconfigure $subframe3_w 1 -minsize 231
    grid $realization_char_w         -row 0 -column 0 -sticky w -pady 0
    grid $distribution_value_combo_w -row 0 -column 1 -sticky e -pady 0
    bind [$realization_char_w component entry]  <KeyRelease> {number_output_files}
 

    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]

    set outfl_w [iwidgets::entryfield $subframe1_w.outfl \
		     -labeltext "ASM, PMWIN file/s  " \
		     -width 57 \
		     -textvariable asmpar(outfl)]

    set outfl_browse_w [iwidgets::pushbutton $subframe1_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".out" \
						   -filetypes {{{Output Files} {.out}} {{All Files} {.*}}} \
						   -initialfile $asmpar(outfl) \
						   -initialdir $exedir \
						   -title "Save EXPORT ASM, PMWIN file as ..."]

				if { $tmp_fname != {} } {
				    set asmpar(outfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $output_frame_w.subfr2]

    set number_outfl_label_w [iwidgets::labeledwidget $subframe2_w.label_number_outfl \
				  -labeltext "Number of output files "]
    number_output_files
    set number_outfl_value_w [iwidgets::labeledwidget $subframe2_w.value_number_outfl \
				  -labelvariable asmpar(noutfl)]
    set outfl_sname_w [iwidgets::labeledwidget $subframe2_w.sname_outfl \
				  -labelvariable asmpar(sname)]


    grid columnconfigure $subframe2_w 0 -minsize 47
    grid columnconfigure $subframe2_w 1 -minsize 45
    grid columnconfigure $subframe2_w 2 -minsize 200
    grid $number_outfl_label_w -row 0 -column 0 -sticky e -pady 0
    grid $number_outfl_value_w -row 0 -column 1 -sticky w -pady 0
    grid $outfl_sname_w        -row 0 -column 2 -sticky w -pady 0

    
    pack \
	$subframe1_w \
	$subframe2_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    pack \
	$data_iframe_w \
	$characterization_iframe_w \
	$output_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $export_asm_box_w.export_asm_tbn view "Files"    
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    
    set button_frame_w [frame $export_asm_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $asmpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select EXPORT ASM, PMWIN parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_export_asm_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       write_export_asm_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { $asmpar(vartype) == "Categorical" } {
				      if { $asmpar(datafl) == {*.out} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file first." \
					      -type ok
				      } elseif { $asmpar(datafl) != {*.out} } {
					  if { $asmpar(charfl) == {*.gtd} } {
					      tk_messageBox \
						  -title "Defaults" \
						  -icon warning \
						  -message "You should select a characterization input file first." \
						  -type ok
					  } elseif { $asmpar(charfl) != {*.gtd} } {
					      if { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
						  tk_messageBox \
						      -title "Characterization Input" \
						      -icon warning \
						      -message "The variables characterization file has to few data.\nYou should select another file." \
						      -type ok
					      } elseif { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
						  set tmp_fname [tk_getSaveFile \
								     -initialdir [file dirname $asmpar(datafl)] \
								     -defaultextension ".par" \
								     -filetypes {{{Parameter Files} {.par}} \
										     {{All Files} {.*}}}\
								     -title "Save EXPORT ASM, PMWIN parameter file as ..."]
						  
						  $save_button_w configure -state normal
						  
						  if { $tmp_fname != {} } {
						      set param_fname $tmp_fname
						      write_export_asm_params $param_fname
						  }
					      }
					  }
				      }
				  } elseif { $asmpar(vartype) == "Continous" } {
				      if { $asmpar(datafl) == {*.out} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a simulation input file first." \
					      -type ok
				      } elseif { $asmpar(datafl) != {*.out} } {
					  if { $asmpar(charfl) != {*.gtd} } {
					      if { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
						  tk_messageBox \
						      -title "Characterization Input" \
						      -icon warning \
						      -message "The variables characterization file has to few data, although the data are not used actually.\nYou should select another file." \
						      -type ok
					      } elseif { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
						  set tmp_fname [tk_getSaveFile \
								     -initialdir [file dirname $asmpar(datafl)] \
								     -defaultextension ".par" \
								     -filetypes {{{Parameter Files} {.par}} \
										     {{All Files} {.*}}}\
								     -title "Save EXPORT ASM, PMWIN parameter file as ..."]
						  
						  $save_button_w configure -state normal
						  
						  if { $tmp_fname != {} } {
						      set param_fname $tmp_fname
						      write_export_asm_params $param_fname
						  }
					      }
					  } elseif { $asmpar(charfl) == {*.gtd} } {
					      set tmp_fname [tk_getSaveFile \
								 -initialdir [file dirname $asmpar(datafl)] \
								 -defaultextension ".par" \
								 -filetypes {{{Parameter Files} {.par}} \
										 {{All Files} {.*}}}\
								 -title "Save EXPORT ASM, PMWIN parameter file as ..."]
					      
					      $save_button_w configure -state normal
					      
					      if { $tmp_fname != {} } {
						  set param_fname $tmp_fname
						  write_export_asm_params $param_fname
					      }
					  }
				      }
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {

			      ### set the cursor on watch
			      . configure -cursor watch
			      update
			      
			      ### message, if there are no data
			      if { $asmpar(datafl) == {*.out} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select a simulation input file first." \
				      -type ok
			      } elseif { $asmpar(datafl) != {*.out} } { ;# ok, there are data
				      
				  ### assignment of characterization data depending on the variable type
				  if { $asmpar(vartype) == "Continous" } {
				      
				      ### test, if the characterization file has enough data
				      if { $asmpar(charfl) != {*.gtd} } {
					  if { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
					      tk_messageBox \
						  -title "Characterization Input" \
						  -icon warning \
						  -message "The variables characterization file has to few data, although the data are not used actually.\nYou should select another file." \
						  -type ok
					  } elseif { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
					      
					      ### message, if the data are not 2D or 3D
					      if { ($asmpar(nx) == 1 && $asmpar(ny) == 1) || \
						       ($asmpar(nx) == 1 && $asmpar(nz) == 1) || \
						       ($asmpar(ny) == 1 && $asmpar(nz) == 1) } {
						  tk_messageBox \
						      -title "Dimension of input data" \
						      -icon error \
						      -message "The input data are not 2D or 3D.\nYou should select another input data file."
					      } elseif { ($asmpar(nx) != 1 && $asmpar(ny) != 1) || \
							     ($asmpar(nx) != 1 && $asmpar(nz) != 1) || \
							     ($asmpar(ny) != 1 && $asmpar(nz) != 1) } {

						  ### write the temporary parameter file
						  set paramfile "tmprun.par"
						  write_export_asm_params_run $paramfile
						  update

						  ### restruct data from the simulated realizations file
						  log_newline
						  log_newline
						  log "Restructuring:\n"
						  log "Restructuring of simulated values."
						  geossav_convert $asmpar(datafl) "confl.out" \
						      $asmpar(nx) $asmpar(ny) $asmpar(nz) $asmpar(nsim)
						  update

						  ### open the simulated realizations file
						  if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
						      tk_messageBox \
							  -title "File open error" \
							  -icon error \
							  -message "The file $fname couldn't open!\n$err_msg"
						  }
						  
						  ### message, if the output file/s cannot be open
						  log_newline
						  log_newline
						  log "Exporting data:\n"
						  if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
						      for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
							  if { [catch {set outf \
									   [open "$asmpar(outfl)_s$iv" {w}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  export_asm_continous_variables_2d $iv
							  log "Exporting of data: simulation $iv"
							  update
							  close $outf
						      }
						  } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
						      if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
							  for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
							      if { [catch {set outf [open "$asmpar(outfl)_s$iv" {w}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      export_asm_continous_variables_2d $iv
							      log "Exporting of data: simulation $iv"
							      update
							      close $outf
							  }
						      } elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
							  for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
							      for { set v 1 } { $v <= $asmpar(nz) } { incr v } {
								  if { [catch {set outfcon3d($iv,$v) [open "$asmpar(outfl)_s$iv,_l$v" {w}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  export_asm_continous_variables_3d $iv $v
								  log "Exporting of data: simulation $iv, layer $v"
								  update
								  close $outfcon3d($iv,$v)
							      }
							  }
						      }
						  }
						  close $inf
						  log_newline
						  log_newline
						  log "EXPORT ASM, PMWIN finished.\n\n"
						  
						  ### delete the files/temporary files
						  catch {
						      file delete -force \
							  $paramfile \
							  "confl.out"
						  }
						  set private(button) 1
					      }
					  }
				      } elseif { $asmpar(charfl) == {*.gtd} } {
					  
					  ### message, if the data are not 2D or 3D
					  if { ($asmpar(nx) == 1 && $asmpar(ny) == 1) || \
						   ($asmpar(nx) == 1 && $asmpar(nz) == 1) || \
						   ($asmpar(ny) == 1 && $asmpar(nz) == 1) } {
					      tk_messageBox \
						  -title "Dimension of input data" \
						  -icon error \
						  -message "The input data are not 2D or 3D.\n You should select another input data file."
					  } elseif { ($asmpar(nx) != 1 && $asmpar(ny) != 1) || \
							 ($asmpar(nx) != 1 && $asmpar(nz) != 1) || \
							 ($asmpar(ny) != 1 && $asmpar(nz) != 1) } {
					      
					      ### write the temporary parameter file
					      set paramfile "tmprun.par"
					      write_export_asm_params_run $paramfile
					      update

					      ### restruct data from the simulated realizations file
					      log_newline
					      log_newline
					      log "Restructuring:\n"
					      log "Restructuring of simulated values."
					      geossav_convert $asmpar(datafl) "confl.out" \
						  $asmpar(nx) $asmpar(ny) $asmpar(nz) $asmpar(nsim)
					      update
					      
					      ### open the simulated realizations file
					      if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
						  tk_messageBox \
						      -title "File open error" \
						      -icon error \
						      -message "The file $fname couldn't open!\n$err_msg"
					      }
					      
					      ### message, if the output file/s cannot be open
					      log_newline
					      log_newline
					      log "Exporting data:\n"
					      if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
						  for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
						      if { [catch {set outf \
								       [open "$asmpar(outfl)_s$iv" {w}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      export_asm_continous_variables_2d $iv
						      log "Exporting of data: simulation $iv"
						      update
						      close $outf
						  }
					      } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horizontal data
						  if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
						      for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
							  if { [catch {set outf [open "$asmpar(outfl)_s$iv" {w}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  export_asm_continous_variables_2d $iv
							  log "Exporting of data: simulation $iv"
							  update
							  close $outf
						      }
						  } elseif  { $asmpar(nz) != 1 } { ;# 3D horizontal data
						      for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
							  for { set v 1 } { $v <= $asmpar(nz) } { incr v } {
							      if { [catch {set outfcon3d($iv,$v) [open "$asmpar(outfl)_s$iv,_l$v" {w}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      export_asm_continous_variables_3d $iv $v
							      log "Exporting of data: simulation $iv, layer $v"
							      update
							      close $outfcon3d($iv,$v)
							  }
						      }
						  }
					      }
					      close $inf
					      log_newline
					      log_newline
					      log "EXPORT ASM, PMWIN finished.\n\n"
					      
					      ### delete the files/temporary files
					      catch {
						  file delete -force \
						      $paramfile \
						      "confl.out"
					      }
					  }
					  set private(button) 1
				      }
				  } elseif { $asmpar(vartype) == "Categorical" } {
				      
				      ### message, if there is no characterization input file
				      if { $asmpar(charfl) == {*.gtd} } {
					  tk_messageBox \
					      -title "Defaults" \
					      -icon warning \
					      -message "You should select a characterization input file first." \
					      -type ok
				      } elseif { $asmpar(charfl) != {*.gtd} } {
					  ### message, if the characterization file has to few data
					  if { $asmpar(charvar) < [expr $asmpar(ncat) + 1] } {
					      tk_messageBox \
						  -title "Characterization Input" \
						  -icon warning \
						  -message "The variables characterization file has to few data.\nYou should select another file." \
						  -type ok
					  } elseif { $asmpar(charvar) >= [expr $asmpar(ncat) + 1] } {
					      
					      ### message, if the data are not 2D or 3D
					      if { ($asmpar(nx) == 1 && $asmpar(ny) == 1) || \
						       ($asmpar(nx) == 1 && $asmpar(nz) == 1) || \
						       ($asmpar(ny) == 1 && $asmpar(nz) == 1) } {
						  tk_messageBox \
						      -title "Dimension of input data" \
						      -icon error \
						      -message "The input data are not 2D or 3D.\nYou should select another input data file."
					      } elseif { ($asmpar(nx) != 1 && $asmpar(ny) != 1) || \
							     ($asmpar(nx) != 1 && $asmpar(nz) != 1) || \
							     ($asmpar(ny) != 1 && $asmpar(nz) != 1) } {
						  
						  ### replace the categorical variables with the mean values of a parameter
						  if { $asmpar(mean) == 1 } {
						      
						      ### write the temporary parameter file
						      set paramfile "tmprun.par"
						      write_export_asm_params_run $paramfile
						      update
						      
						      ### restruct data from the simulated realizations file
						      log_newline
						      log_newline
						      log "Restructuring:\n"
						      log "Restructuring of simulated categories."
						      geossav_convert $asmpar(datafl) "confl.out" \
							  $asmpar(nx) $asmpar(ny) $asmpar(nz) $asmpar(nsim)
						      update

						      ### open the simulated realizations file
						      if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      
						      ### open the variable characterization file
						      if { [catch {set charf [open $asmpar(charfl) {r}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      
						      ### open the replacement file and replace the categorical 
						      ### variables with mean values
						      log_newline
						      log_newline
						      log "Characterization:\n"
						      log "Characterization of simulated categories with mean values."
						      update
						      if { [catch {set repf [open "repfl.out" {w+}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      mean_value_replacement
						      close $inf
						      close $charf
						      
						      ### message, if the output file/s cannot be open
						      log_newline
						      log_newline
						      log "Exporting of data:\n"
						      if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
							  for { set i 1 } { $i <= $asmpar(nsim) } { incr i } {
							      ;# in that case nsim = noutfl
							      if { [catch {set outf [open "$asmpar(outfl)_s$i" {w}]} err_msg] } {
								  tk_messageBox \
								      -title "File open error" \
								      -icon error \
								      -message "The file $fname couldn't open!\n$err_msg"
							      }
							      export_asm_categorical_variables_mean_2d $i
							      log "Exporting of data: simulation $i, characterization 1"
							      update
							      close $outf
							  }
						      } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horiz. data
							  if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
							      for { set i 1 } { $i <= $asmpar(nsim) } { incr i } {
								  if { [catch {set outf [open "$asmpar(outfl)_s$i" {w}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  export_asm_categorical_variables_mean_2d $i
								  log "Exporting of data: simulation $i, characterization 1"
								  update
								  close $outf
							      }
							  } elseif { $asmpar(nz) != 1 } { ;# 3D horizontal data
							      for { set iv 1 } { $iv <= $asmpar(nsim) } { incr iv } {
								  for { set v 1 } { $v <= $asmpar(nz) } { incr v } {
								      if { [catch {set outfcatm3d($iv,$v) [open "$asmpar(outfl)_s$iv,_l$v,_c1" {w}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								      export_asm_categorical_variables_mean_3d $iv $v
								      log "Exporting of data: simulation $iv, layer $v, characterization 1"
								      update
								      close $outfcatm3d($iv,$v)
								  }
							      }
							  }
						      }
						      log_newline
						      log_newline
						      log "EXPORT ASM, PMWIN finished.\n\n"
						      close $repf
						      
						      ### delete the files/temporary files
						      catch {
							  file delete -force \
							      $paramfile \
							      "confl.out" \
							      "repfl.out"
						      }
						  } elseif { $asmpar(mean) == 0 } { 
						      ;# replace the categorical variables with values from 
						      ;# the distribution of a parameter
						  
						      ### write the temporary parameter file
						      set paramfile "tmprun.par"
						      write_export_asm_params_run $paramfile
						      update
						      
						      ### restruct data from the simulated realizations file
						      log_newline
						      log_newline
						      log "Restructuring:\n"
						      log "Restructuring of simulated categories."
						      geossav_convert $asmpar(datafl) "confl.out" \
							  $asmpar(nx) $asmpar(ny) $asmpar(nz) $asmpar(nsim)
						      update

						      ### open the simulated realizations file
						      if { [catch {set inf [open "confl.out" {r}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      
						      ### open the variable characterization file
						      if { [catch {set charf [open $asmpar(charfl) {r}]} err_msg] } {
							  tk_messageBox \
							      -title "File open error" \
							      -icon error \
							      -message "The file $fname couldn't open!\n$err_msg"
						      }
						      
						      ### open the replacement files and replace the categorical 
						      ### variables with values from the distribution
						      log_newline
						      log_newline
						      log "Characterization:\n"
						      log "Characterization of simulated categories with values from the distributions."
						      update
						      for { set i 1 } { $i <= $asmpar(nsimchar) } { incr i } {
							  if { [catch {set repfd($i) [open "repfl.out$i" {w+}]} err_msg] } {
							      tk_messageBox \
								  -title "File open error" \
								  -icon error \
								  -message "The file $fname couldn't open!\n$err_msg"
							  }
							  distribution_value_replacement $i
						      }
						      close $inf
						      close $charf

						      ### message, if the output file/s cannot be open
						      log_newline
						      log_newline
						      log "Exporting data:\n"
						      if { $asmpar(nx) == 1 || $asmpar(ny) == 1 } { ;# 2D vertical data
							  for { set vi 1 } { $vi <= $asmpar(nsimchar) } { incr vi }  {
							      for { set i 1 } { $i <= $asmpar(nsim) } { incr i } {
								  if { [catch {set outfcatd2d($i,$vi) [open "$asmpar(outfl)_s$i,_c$vi" {w}]} err_msg] } {
								      tk_messageBox \
									  -title "File open error" \
									  -icon error \
									  -message "The file $fname couldn't open!\n$err_msg"
								  }
								  export_asm_categorical_variables_distribution_2d $i $vi
								  log "Exporting of data: simulation $i, characterization $vi"
								  update
								  close $outfcatd2d($i,$vi)
							      }
							      close $repfd($vi)
							  }
						      } elseif { $asmpar(nx) != 1 && $asmpar(ny) != 1 } { ;# horiz. data
							  if  { $asmpar(nz) == 1 } { ;# 2D horizontal data
							      for { set vi 1 } { $vi <= $asmpar(nsimchar) } { incr vi }  {
								  for { set i 1 } { $i <= $asmpar(nsim) } { incr i } {
								      if { [catch {set outfcatd2d($i,$vi) [open "$asmpar(outfl)_s$i,_c$vi" {w}]} err_msg] } {
									  tk_messageBox \
									      -title "File open error" \
									      -icon error \
									      -message "The file $fname couldn't open!\n$err_msg"
								      }
								      export_asm_categorical_variables_distribution_2d $i $vi
								      log "Exporting of data: simulation $i, characterization $vi"
								      update
								      close $outfcatd2d($i,$vi)
								  }
								  close $repfd($vi)
							      }
							  } elseif { $asmpar(nz) != 1 } { ;# 3D horizontal data
							      for { set vi 1 } { $vi <= $asmpar(nsimchar) } { incr vi } {
								  for { set i 1 } { $i <= $asmpar(nsim) } { incr i } {
								      for { set v 1 } { $v <= $asmpar(nz) } { incr v } {
									  if { [catch {set outfcatd3d($i,$v,$vi) [open "$asmpar(outfl)_s$i,_l$v,_c$vi" {w}]} err_msg] } {
									      tk_messageBox \
										  -title "File open error" \
										  -icon error \
										  -message "The file $fname couldn't open!\n$err_msg"
									  }
									  export_asm_categorical_variables_distribution_3d $i $v $vi
									  log "Exporting of data: simulation $i, layer $v, characterization $vi"
									  update
									  close $outfcatd3d($i,$v,$vi)
								      }
								  }
								  close $repfd($vi)
							      }
							  }
						      }
						      log_newline
						      log_newline
						      log "EXPORT ASM, PMWIN finished.\n\n"
						      
						      ### delete the files/temporary files
						      catch {
							  file delete -force \
							      $paramfile \
							      "confl.out"
						      }
						      for { set i 1 } { $i <= $asmpar(nsimchar) } { incr i } {
							  catch {
							      file delete -force "repfl.out$i"
							      update
							  }
						      }
						  }
						  set private(button) 1  
					      }
					  }
				      }
				  }
			      }
			      ### reset the cursor on arrow 
			      . configure -cursor arrow
			      update
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $export_asm_box_w
    
#    tkwait visibility $export_asm_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $export_asm_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $export_asm_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    update
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
	
	bind $export_asm_box_w <Destroy> {}
	destroy $export_asm_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }   
    return
}
#============================================================================
#=END=OF=EXPORT_ASM_PMWIN====================================================
