##START#OF#GAMV##############################################################
# geossav_gamv.tcl
#
# Source script creating the gamv tabnotebook in GEOSSAV,
# developed at the University of Basel, Switzerland,
# by Christian Regli and Lukas Rosenthaler
#
#############################################################################


#############################################################################
#
# The following set of parameters are required for the GAMV
#
global exedir
source $exedir/Geossav_gamv_globals.tcl

set gamvpar(datafl) "*.dat"              ;# input data file, simplified Geo-EAS format
set gamvpar(icolx) 1                     ;# column number for the x coordinates
set gamvpar(icoly) 2                     ;# column number for the y coordinates, icoly=0 and icolz=0 
                                         ;# indicates 1-D simulation
set gamvpar(icolz) 3                     ;# column number for the z coordinates, icolz=0 indicates 
                                         ;# 2-D simulation
set gamvpar(nvar) 1                      ;# number of variables to consider
set gamvpar(ivar) 1                      ;# column order of the variables
set gamvpar(tmin) -1e+21                 ;# minimum trimming limit, all values strictly less than tmin 
                                         ;# are ignored
set gamvpar(tmax) +1e+21                 ;# maximum trimming limit, all values strictly greater or equal than 
                                         ;# tmax are ignored
set gamvpar(outfl) "*.out"               ;# output data file containing the results: variograms ordered by direction
                                         ;# and then variogram type specified in the parameter file (the directions
                                         ;# cycle fastest then the variogram number); for each variogram there is a 
                                         ;# one-line description and then nlag lines each with the following:
                                         ;# 1. lag number (increasing from 1 to nlag)
                                         ;# 2. average separation distance for the lag
                                         ;# 3. semivariogram value (whatever type was specivied)
                                         ;# 4. number of pairs for the lag
                                         ;# 5. mean of the data contributing to the tail
                                         ;# 6. mean of the data contributing to the head
                                         ;# 7. tail and head variances (for the correlogram)
set gamvpar(nlag) 10                     ;# number of lags to compute (same for all directions)
set gamvpar(xlag) 10                     ;# unit lag separation distance
set gamvpar(xltol) 5                     ;# lag tolerance, if sltol is entered as zero, it will be reset to xlag/2
set gamvpar(ndir) 1                      ;# number of directions to consider
set gamvpar(azm) 0                       ;# azimuth angle
set gamvpar(atol) 90                     ;# half-window azimuth tolerance 
set gamvpar(bandwh) 1000                 ;# azimuth bandwidth
set gamvpar(dip) 0                       ;# dip angle
set gamvpar(dtol) 90                     ;# half-window dip tolerance 
set gamvpar(bandwd) 1000                 ;# dip bandwidth
set gamvpar(standardize) 1               ;# if set to 1, the semivariogram values will be devided by the variance
set gamvpar(nvarg) 1                     ;# number of variograms to consider
set gamvpar(title) 0                     ;# flag for writing parameter title
#
# The following set of parameters are required for each of the nvarg variograms
#
set gamvpar(ivtail) 1                    ;# variable for the tail, for direct variograms the ivtail array is identical
                                         ;# to the ivhead array 
set gamvpar(ivhead) 1                    ;# variable for the head, for direct variograms the ivtail array is identical
                                         ;# to the ivhead array 
set gamvpar(ivtype) 1                    ;# type of variogram to b computed
set gamvpar(cut) 1                       ;# whenever the ivtype is set to 9 or 10 (indicator variogram) a cutoff must
                                         ;# be specified 
#
# The following set of parameters are required for GEOSSAV GAMV, especially for the feeding of the 
# GAMV parameters tabnotebook after running GAMV (GAMV parameters tabnotebook disappear after 
# start GAMV) or after cancel and restart the GAMV parameters tabnotebook without an exit of Geossav
#
set gamvpar(variables) 4
set gamvpar(cancel_counter) 0
#
# End of initialization of variables. This has to be done once at the start 
# of the programm.
#
#############################################################################


#############################################################################
#
# Start of proc section.
#

#############################################################################
proc gamv_read_data { fname } {
    global gamvpar
    global dim_name_data

    lappend dim_name_data "None"
    if { [file readable $fname] } {
	set inf [open $fname "r"]
	gets $inf
	gets $inf n
	for { set i 0 } { $i < $n } { incr i } {
	    gets $inf dum
	    lappend dim_name_data $dum
	    set tmp_parameters [expr $i + 1]
	}
	close $inf

	if { $gamvpar(cancel_counter) == 0 } {
	    set gamvpar(variables) $tmp_parameters
	} elseif { $gamvpar(cancel_counter) != 0 } {
	    set gamvpar(variables) $gamvpar(variables)
	}
	
	return 1
    } else {
	return 0
    }
}
#============================================================================


#############################################################################
proc regexpvalidate { char } {
    return [regexp {![ ]} $char]
}
#============================================================================


#############################################################################
proc plus_variables { } {
    global gamvpar
    global ivar_vals
    global ivar_table_w
    global dim_name_data
    global logwin
    
    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Variables" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(nvar) < 4 } {

	    set gamvpar(nvar) [expr $gamvpar(nvar) + 1]
	    set ii $gamvpar(nvar) 

	    ### add and fill an additional row in the variables table
	    $ivar_table_w configure -rows [expr $ii + 1]
	    
	    set ivar_vals($ii,label) [iwidgets::labeledwidget $ivar_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    update
	    
	    set ivar_vals($ii,combo) [iwidgets::combobox $ivar_table_w.combo$ii \
					  -textvariable gamvpar(ivar,$ii) \
					  -validate {regexpvalidate %c} \
					  -listheight 60 \
					  -width 29]
	    
	    foreach i $dim_name_data {
		$ivar_vals($ii,combo) insert list end $i
		$ivar_vals($ii,combo) delete list [expr $gamvpar(variables) + 1]
	    }
	    $ivar_vals($ii,combo) delete list 0
	    if { $gamvpar(variables) >= 7 } {
		$ivar_vals($ii,combo) selection set [expr $ii + 2]
		set gamvpar(ivar) [lindex $dim_name_data [expr $ii + 3]]
	    } elseif { $gamvpar(variables) < 7 && $gamvpar(variables) >= 4 } {
		$ivar_vals($ii,combo) selection set 3
		set gamvpar(ivar) [lindex $dim_name_data 3]
	    } elseif { $gamvpar(variables) < 4 } {
		$ivar_vals($ii,combo) selection set 0
		set gamvpar(ivar) [lindex $dim_name_data 0]
	    }
	    update
	    
	    $ivar_table_w window configure $ii,0 -window $ivar_vals($ii,label) -relief flat -background gray50
	    $ivar_table_w window configure $ii,1 -window $ivar_vals($ii,combo)

	} elseif { $gamvpar(nvar) >= 4 } {
	    tk_messageBox \
		-title "Variables" \
		-icon warning \
		-message "Maximum 4 variables are permitted in a single pass." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc minus_variables { } {
    global gamvpar
    global ivar_vals
    global ivar_table_w
    global logwin
    
    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Variables" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(nvar) > 1 } {

	    set ii $gamvpar(nvar)

	    ### destroy the last row in the variables table
	    destroy $ivar_vals($ii,label)
	    destroy $ivar_vals($ii,combo)
	    $ivar_table_w configure -rows $ii

	    set gamvpar(nvar) [expr $gamvpar(nvar) - 1]

	} elseif { $gamvpar(nvar) <= 1 } {
	    tk_messageBox \
		-title "Variables" \
		-icon warning \
		-message "At least one variable is needed in a single pass." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc check_number_t { entry newstr } {
    return [regexp {^[+-]?[0-9]*[\.]?[0-9]*([e][+-]?[0-9]*)?$} $newstr]
}
#============================================================================


#############################################################################
proc check_number_nlag { entry newstr char } {
    set current [$entry get]
    set len [string length $current]
    if { $len < 2 } {
	return [regexp {^[0-9]*$} $newstr]
    } elseif { $len == 2 && [string match {[0-9]} $char] } {
	$entry delete 0 end  
	$entry insert end "100"
    }
}
#============================================================================


#############################################################################
proc check_separation_tolerance { entry newstr char } {
    set current [$entry get]
    if {$current == {} && [string match {[0-9.]} $char]} {
	return [regexp {[0-9.]} $char]
    } elseif {$current != {} && [string match {[0-9.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    }
}
#============================================================================


#############################################################################
proc plus_var_dir { } {
    global gamvpar
    global vario_dir_table_w
    global vario_dir_vals
    global vario_dir_azimuth
    global vario_dir_az_tol
    global vario_dir_bandwidth_h
    global vario_dir_dip
    global vario_dir_dip_tol
    global vario_dir_bandwidth_v

    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Number of variogram directions" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(ndir) < 4 } {
	    
	    set gamvpar(ndir) [expr $gamvpar(ndir) + 1]
	    set ii $gamvpar(ndir)

	    ### add an additional row in the vario_dir table
	    $vario_dir_table_w configure -rows [expr $gamvpar(ndir) + 1]
	    set vario_dir_vals($ii,label) [iwidgets::labeledwidget $vario_dir_table_w.vals$ii \
					       -labeltext $gamvpar(ndir) \
					       -background gray50 \
					       -foreground white]
	    update
	    
	    set gamvpar(azm,$ii) 0
	    set vario_dir_azimuth($ii,entry) [iwidgets::entryfield $vario_dir_table_w.azimuth$ii \
						  -textvariable gamvpar(azm,$ii) \
						  -validate {check_number_vario_dir_azimuth %W %P %c}]
	    update
	    
	    set gamvpar(atol,$ii) 90
	    set vario_dir_az_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.az_tol$ii \
						 -textvariable gamvpar(atol,$ii) \
						 -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    update
	    
	    set gamvpar(bandwh,$ii) 100
	    set vario_dir_bandwidth_h($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_h$ii \
						      -textvariable gamvpar(bandwh,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    update
	    
	    set gamvpar(dip,$ii) 0
	    set vario_dir_dip($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip$ii \
					      -textvariable gamvpar(dip,$ii) \
					      -validate {check_number_vario_dir_dip %W %P %c}]
	    update
	    
	    set gamvpar(dtol,$ii) 90
	    set vario_dir_dip_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip_tol$ii \
						  -textvariable gamvpar(dtol,$ii) \
						  -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    update
	    
	    set gamvpar(bandwd,$ii) 100
	    set vario_dir_bandwidth_v($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_v$ii \
						      -textvariable gamvpar(bandwd,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    update
	    
	    $vario_dir_table_w window configure $ii,0 -window $vario_dir_vals($ii,label) -relief flat -background gray50
	    $vario_dir_table_w window configure $ii,1 -window $vario_dir_azimuth($ii,entry)
	    $vario_dir_table_w window configure $ii,2 -window $vario_dir_az_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,3 -window $vario_dir_bandwidth_h($ii,entry)
	    $vario_dir_table_w window configure $ii,4 -window $vario_dir_dip($ii,entry)
	    $vario_dir_table_w window configure $ii,5 -window $vario_dir_dip_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,6 -window $vario_dir_bandwidth_v($ii,entry)
	    
	} elseif { $gamvpar(ndir) >= 4 } {
	    tk_messageBox \
		-title "Number of variogram directions" \
		-icon warning \
		-message "Maximum 4 variogram directions are permitted in a singel pass." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc minus_var_dir { } {
    global gamvpar
    global vario_dir_table_w
    global vario_dir_vals
    global vario_dir_azimuth
    global vario_dir_az_tol
    global vario_dir_bandwidth_h
    global vario_dir_dip
    global vario_dir_dip_tol
    global vario_dir_bandwidth_v

    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Number of variogram directions" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(ndir) > 1 } {
	    
	    set ii $gamvpar(ndir)
	    
	    ### destroy the last row in the vario_dir table
	    destroy $vario_dir_vals($ii,label)
	    destroy $vario_dir_azimuth($ii,entry)
	    destroy $vario_dir_az_tol($ii,entry)
	    destroy $vario_dir_bandwidth_h($ii,entry)
	    destroy $vario_dir_dip($ii,entry)
	    destroy $vario_dir_dip_tol($ii,entry)
	    destroy $vario_dir_bandwidth_v($ii,entry)
	    $vario_dir_table_w configure -rows $ii
	    
	    set gamvpar(ndir) [expr $gamvpar(ndir) - 1]
	    
	} else {
	    tk_messageBox \
		-title "Number of variogram directions" \
		-icon warning \
		-message "At least 1 variogram direction is needed in a single pass." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc check_number_vario_dir_azimuth { entry newstr char } {
    if {$newstr == {} || ($newstr >= 0 && $newstr <= 360) && [string match {[0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 360 && [string match {[.0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "360"
    }
}
#============================================================================


#############################################################################
proc check_number_vario_dir_az_and_dip_tol { entry newstr char } {
    if {$newstr == {} || ($newstr >= 0 && $newstr <= 90) && [string match {[0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 90 && [string match {[.0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "90"
    }
}
#============================================================================


#############################################################################
proc check_number_vario_dir_bandwidth { entry newstr char } {
    set current [$entry get]
    catch {scan $current {%g} current}
    if {$current < 100} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$current >= 100 && [string match {[0-9.]} $char]} {
	$entry delete 0 end  
	$entry insert end "1000"
    }
}
#============================================================================


#############################################################################
proc check_number_vario_dir_dip { entry newstr char } {
    if {$newstr == {} || ($newstr >= -90 && $newstr <= 90) || $newstr == {-} && [string match {[-0-9.]} $char]} {
	return [regexp {^[-]?[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$newstr > 90 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "90"
    } elseif {$newstr < -90 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "-90"
    }
}
#============================================================================


#############################################################################
proc plus_var_type { } {
    global gamvpar
    global vario_type_table_w
    global vario_type_vals
    global vario_type_tail
    global vario_type_head
    global vario_type_type
    global vario_type_threshold
    global dim_name_data

    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Number of variogram types" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(nvarg) < 4 } {
	    
	    set gamvpar(nvarg) [expr $gamvpar(nvarg) + 1]
	    set ii $gamvpar(nvarg)

	    ### add an additional row in the vario_type table
	    $vario_type_table_w configure -rows [expr $gamvpar(nvarg) + 1]
	    set vario_type_vals($ii,label) [iwidgets::labeledwidget $vario_type_table_w.vals$ii \
						-labeltext "$ii" \
						-background gray50 \
						-foreground white]
	    update			       
	    
	    set gamvpar(ivtail,$ii) 1
	    set vario_type_tail($ii,combo) [iwidgets::combobox $vario_type_table_w.tail$ii \
						-textvariable gamvpar(ivtail,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_tail $ii"]
	    foreach i $dim_name_data {
		$vario_type_tail($ii,combo) insert list end $i
		$vario_type_tail($ii,combo) delete list [expr $gamvpar(variables) + 1]
	    }
	    $vario_type_tail($ii,combo) delete list 0
	    if { $gamvpar(variables) >= 4 } {
		$vario_type_tail($ii,combo) selection set 3
		set gamvpar(ivtail,$ii) [lindex $dim_name_data 4]
	    } elseif { $gamvpar(variables) < 4 } {
		$vario_type_tail($ii,combo) selection set 0
		set gamvpar(ivtail,$ii) [lindex $dim_name_data 1]
	    }
	    update

	    set gamvpar(ivhead,$ii) 1
	    set vario_type_head($ii,combo) [iwidgets::combobox $vario_type_table_w.head$ii \
						-textvariable gamvpar(ivhead,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_head $ii"]
	    foreach i $dim_name_data {
		$vario_type_head($ii,combo) insert list end $i
		$vario_type_head($ii,combo) delete list [expr $gamvpar(variables) + 1]
	    }
	    $vario_type_head($ii,combo) delete list 0
	    if { $gamvpar(variables) >= 4 } {
		$vario_type_head($ii,combo) selection set 3
		set gamvpar(ivhead,$ii) [lindex $dim_name_data 4]
	    } elseif { $gamvpar(variables) < 4 } {
		$vario_type_head($ii,combo) selection set 0
		set gamvpar(ivhead,$ii) [lindex $dim_name_data 1]
	    }
	    update
	    
	    set gamvpar(ivtype,$ii) 0
	    set vario_type_type($ii,combo) [iwidgets::combobox $vario_type_table_w.type$ii \
						-textvariable gamvpar(ivtype,$ii) \
						-selectioncommand "threshold_activator $ii" \
						-width 40 \
						-listheight 75 \
						-validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if gamvpar(ivtype,$ii) is empty
	    $vario_type_table_w.type$ii insert list end \
		"Traditional semi-variogram" "Traditional cross semi-variogram" "Covariance" "Correlogram" "General relative semivariogram" "Pairwise relative semi-variogram" "Semivariogram of logarithms" "Semi-madogram" "Indicator semi-variogram - continous variable" "Indicator semi-variogram - categorical variable"
	    if { $gamvpar(ivtype,$ii) == "Traditional semi-variogram" } {
		set sel 0	
	    } elseif { $gamvpar(ivtype,$ii) == "Traditional cross semi-variogram" } {
		set sel 1
	    } elseif { $gamvpar(ivtype,$ii) == "Covariance" } {
		set sel 2
	    } elseif { $gamvpar(ivtype,$ii) == "Correlogram" } {
		set sel 3
	    } elseif { $gamvpar(ivtype,$ii) == "General relative semivariogram" } {
		set sel 4
	    } elseif { $gamvpar(ivtype,$ii) == "Pairwise relative semi-variogram" } {
		set sel 5
	    } elseif { $gamvpar(ivtype,$ii) == "Semivariogram of logarithms" } {
		set sel 6
	    } elseif { $gamvpar(ivtype,$ii) == "Semi-madogram" } {
		set sel 7
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - continous variable" } {
		set sel 8
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - categorical variable" } {
		set sel 9
	    }
	    $vario_type_type($ii,combo) selection set $sel
	    set $vario_type_type($ii,combo) [lindex $vario_type_table_w.type$ii $sel]
	    update
	    
	    set gamvpar(cut,$ii) 0
	    set vario_type_threshold($ii,entry) [iwidgets::entryfield $vario_type_table_w.threshold$ii \
						     -textvariable gamvpar(cut,$ii) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate "check_number_vario_type_threshold %W %P %c $ii"]
	    update
	    
	    
	    $vario_type_table_w window configure $ii,0 -window $vario_type_vals($ii,label) -relief flat -background gray50
	    $vario_type_table_w window configure $ii,1 -window $vario_type_tail($ii,combo)
	    $vario_type_table_w window configure $ii,2 -window $vario_type_head($ii,combo)
	    $vario_type_table_w window configure $ii,3 -window $vario_type_type($ii,combo)
	    $vario_type_table_w window configure $ii,4 -window $vario_type_threshold($ii,entry)
	    
	} elseif { $gamvpar(nvarg) >= 4 } {
	    tk_messageBox \
		-title "Number of variogram types" \
		-icon warning \
		-message "Maximum 4 variogram types are permitted in a singel pass." \
		-type ok
	}  
    }
}
#============================================================================


#############################################################################
proc minus_var_type { } {
    global gamvpar
    global vario_type_table_w
    global vario_type_vals
    global vario_type_tail
    global vario_type_head
    global vario_type_type
    global vario_type_threshold

    if { $gamvpar(datafl) == {*.dat} } {
	tk_messageBox \
	    -title "Number of variogram types" \
	    -icon warning \
	    -message "You should select a file first." \
	    -type ok
    }
    if { $gamvpar(datafl) != {*.dat} } {
	if { $gamvpar(nvarg) > 1 } {
	    
	    set ii $gamvpar(nvarg)
	    
	    ### destroy the last row in the vario_type table
	    destroy $vario_type_vals($ii,label)
	    destroy $vario_type_tail($ii,combo)
	    destroy $vario_type_head($ii,combo)
	    destroy $vario_type_type($ii,combo)
	    destroy $vario_type_threshold($ii,entry)
	    $vario_type_table_w configure -rows $ii

	    set gamvpar(nvarg) [expr $gamvpar(nvarg) - 1]
	    
	} else {
	    tk_messageBox \
		-title "Number of variogram types" \
		-icon warning \
		-message "At least 1 variogram type is needed in a single pass." \
		-type ok
	}
    }
}
#============================================================================


#############################################################################
proc check_variable_tail { ii } {
    global gamvpar
    global vario_type_tail
    global ivar_vals
    global listcheck
    global dim_name_data
    global sel
    
    set listcheck 0
    
    for { set i 1 } { $i <= $gamvpar(nvar) } { incr i } {
	if { "[$vario_type_tail($ii,combo) getcurselection]" == "[$ivar_vals($i,combo) getcurselection]" } {
	    set listcheck [expr $listcheck + 0]
	} elseif { "[$vario_type_tail($ii,combo) getcurselection]" != "[$ivar_vals($i,combo) getcurselection]" } {
	    set listcheck [expr $listcheck + 1]
	}
    }
    
    if { $listcheck >= [expr $gamvpar(nvar)] } {
	$vario_type_tail($ii,combo) clear list
	foreach i $dim_name_data {
	    $vario_type_tail($ii,combo) insert list end $i
	}
	$vario_type_tail($ii,combo) delete list 0
	for { set iii 1 } { $iii <= $gamvpar(variables) } { incr iii } {
	    if { [$ivar_vals(1,combo) getcurselection] == [$vario_type_tail($ii,combo) get $iii] } {
		set sel $iii
	    }
	}
	$vario_type_tail($ii,combo) selection set $sel
	set gamvpar(ivtail,$ii) [lindex $dim_name_data [expr $sel + 1]]
	
	tk_messageBox \
	    -title "Tail variable" \
	    -icon warning \
	    -message "The selection of this variable is not allowed.\n You should select a listed variable or enlarge the list of variables." \
	    -type ok
    } elseif { $listcheck < [expr $gamvpar(nvar)] } {
	return 0
    }     
}
#============================================================================


#############################################################################
proc check_variable_head { ii } {
    global gamvpar
    global vario_type_head
    global ivar_vals
    global listcheck
    global dim_name_data
    global sel
    
    set listcheck 0
    
    for { set i 1 } { $i <= $gamvpar(nvar) } { incr i } {
	if { "[$vario_type_head($ii,combo) getcurselection]" == "[$ivar_vals($i,combo) getcurselection] "} {
	    set listcheck [expr $listcheck + 0]
	} elseif { "[$vario_type_head($ii,combo) getcurselection]" != "[$ivar_vals($i,combo) getcurselection]" } {
	    set listcheck [expr $listcheck + 1]
	}
    }
    
    if { $listcheck >= [expr $gamvpar(nvar)] } {
	$vario_type_head($ii,combo) clear list
	foreach i $dim_name_data {
	    $vario_type_head($ii,combo) insert list end $i
	}
	$vario_type_head($ii,combo) delete list 0
	for { set iii 1 } { $iii <= $gamvpar(variables) } { incr iii } {
	    if { [$ivar_vals(1,combo) getcurselection] == [$vario_type_head($ii,combo) get $iii] } {
		set sel $iii
	    }
	}
	$vario_type_head($ii,combo) selection set $sel
	set gamvpar(ivhead,$ii) [lindex $dim_name_data [expr $sel + 1]]
	
	tk_messageBox \
	    -title "Head variable" \
	    -icon warning \
	    -message "The selection of this variable is not allowed.\n You should select a listed variable or enlarge the list of variables." \
	    -type ok
    } elseif { $listcheck < [expr $gamvpar(nvar)] } {
	return 0
    }     
}
#============================================================================


#############################################################################
proc threshold_activator { ii } {
    global gamvpar
    global vario_type_table_w
    global vario_type_type
    
    set gamvpar(nvarg,$ii) [$vario_type_type($ii,combo) curselection]
    
    if { $gamvpar(nvarg,$ii) <= 7 } {
	$vario_type_table_w.threshold$ii configure \
	    -state disabled \
	    -textbackground gray85 \
	    -foreground gray50
    } elseif { $gamvpar(nvarg,$ii) >= 8 } {
	$vario_type_table_w.threshold$ii configure \
	    -state normal \
	    -textbackground white \
	    -foreground black
    }
}
#============================================================================
    

#############################################################################
proc check_number_vario_type_threshold { entry newstr char ii } {
    set current [$entry get]
    catch {scan $current {%g} current}
    if {$current < 10} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    } elseif {$current >= 10 && [string match {[0-9]} $char]} {
	$entry delete 0 end  
	$entry insert end "100"
    } elseif {$current >= 10 && $current <= 100 && [string match {[\.]} $char]} {
	return [regexp {^[0-9]*[\.]?[0-9]*$} $newstr]
    }
}
#============================================================================


#############################################################################
proc read_gamv_params { fname } {
    global gamvpar
    global dim_name_data
    global icolx_w
    global icoly_w
    global icolz_w
    global ivar_table_w
    global vario_dir_table_w
    global vario_type_table_w
    global ivar_vals
    global vario_dir_vals
    global vario_dir_azimuth
    global vario_dir_az_tol
    global vario_dir_bandwidth_h
    global vario_dir_dip
    global vario_dir_dip_tol
    global vario_dir_bandwidth_v
    global vario_type_vals
    global vario_type_tail
    global vario_type_head
    global vario_type_type
    global vario_type_threshold
    global gamv_box_w
    global datafl_w
    
    ### message, if file cannot be read
    if { [catch {set inf [open $fname {r}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }
    
    ### sets the cursor on watch as long as parameters are reading
    . configure -cursor watch
    update

    ### searches the line "parameters for gamv" and reject all preceded lines
    set line [gets $inf]
    set param_line [string trim $line]
    while { $param_line != "Parameters for GAMV" } {
	set line [gets $inf]
	set param_line [string trim $line]
	if { [eof $inf] } {
	    tk_messageBox \
		-title "File selection error" \
		-icon error \
		-message "You have selected a wrong parameter file.\n"
	    return 0
	}
    }

    ### set file name in gamv box widget
    set param_dname [file dirname $fname]
    set dname_length [string length $param_dname]
    set fname_short [string range $fname [expr $dname_length + 1] end]
    wm title $gamv_box_w "GAMV parameters   -   $fname_short"
    
    ### searches the line "start of parameters:" and reject all preceded lines
    set line [gets $inf]
    while { $line != "START OF PARAMETERS:" } {
	set line [gets $inf]
    }

    ### destroys the content of the ivar-table, the vario_dir_table and the vario_type_table 
    if { $gamvpar(datafl) != {*.dat} } {
	for { set ii 1 } { $ii <= $gamvpar(nvar) } { incr ii } {
	    destroy $ivar_vals($ii,label)
	    destroy $ivar_vals($ii,combo)
	}
	for { set ii 1 } { $ii <= $gamvpar(ndir) } { incr ii } {
	    destroy $vario_dir_vals($ii,label)
	    destroy $vario_dir_azimuth($ii,entry)
	    destroy $vario_dir_az_tol($ii,entry)
	    destroy $vario_dir_bandwidth_h($ii,entry)
	    destroy $vario_dir_dip($ii,entry)
	    destroy $vario_dir_dip_tol($ii,entry)
	    destroy $vario_dir_bandwidth_v($ii,entry)
	}
	for { set ii 1 } { $ii <= $gamvpar(nvarg) } { incr ii } {
	    destroy $vario_type_vals($ii,label)
	    destroy $vario_type_tail($ii,combo)
	    destroy $vario_type_head($ii,combo)
	    destroy $vario_type_type($ii,combo)
	    destroy $vario_type_threshold($ii,entry)
	}
    } elseif { $gamvpar(datafl) == {*.dat} } {
	for { set ii 1 } { $ii <= 1} { incr ii } {
	    destroy $ivar_vals($ii,label)
	    destroy $ivar_vals($ii,combo)
	    
	    destroy $vario_dir_vals($ii,label)
	    destroy $vario_dir_azimuth($ii,entry)
	    destroy $vario_dir_az_tol($ii,entry)
	    destroy $vario_dir_bandwidth_h($ii,entry)
	    destroy $vario_dir_dip($ii,entry)
	    destroy $vario_dir_dip_tol($ii,entry)
	    destroy $vario_dir_bandwidth_v($ii,entry)
	    
	    destroy $vario_type_vals($ii,label)
	    destroy $vario_type_tail($ii,combo)
	    destroy $vario_type_head($ii,combo)
	    destroy $vario_type_type($ii,combo)
	    destroy $vario_type_threshold($ii,entry)
	}
    }    
    
    ### reads name of datafl: file with hard data
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set gamvpar(datafl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set gamvpar(datafl) $line
    }

    if { $gamvpar(datafl) == {*.dat} } {
	$datafl_w configure -state disabled -textbackground gray85 -foreground gray50
    } elseif { $gamvpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }

    ### reads the icolx, ..y, ..z: columns for x, y, z, and fills the icol.. comboboxes
    set line [gets $inf]
    set space [string first " " $line]
    set tmp_gamvpar(icolx) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_gamvpar(icoly) [format "%d" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set tmp_gamvpar(icolz) [format "%d" [string range $line 0 [expr $space - 1]]]
    
    unset dim_name_data
    if { [gamv_read_data $gamvpar(datafl)] != 1 } {
	set dim_name_data {}
	lappend dim_name_data "None"
	for { set i 1 } { $i < 4 } { incr i } {
	    lappend dim_name_data $i
	}
    }

    $icolx_w clear list
    foreach i $dim_name_data {
	$icolx_w insert list end $i
    }
    $icolx_w selection set $tmp_gamvpar(icolx)
    
    $icoly_w clear list
    foreach i $dim_name_data {
	$icoly_w insert list end $i
    }
    $icoly_w selection set $tmp_gamvpar(icoly)
    
    $icolz_w clear list
    foreach i $dim_name_data {
	$icolz_w insert list end $i
    }
    $icolz_w selection set $tmp_gamvpar(icolz)

    ### reads the nvar and ivar(1)...ivar(nvar): columns for ivar and fills the nvar... comboboxes
    set line [gets $inf]
    set varline $line
    set space [string first " " $varline]
    set gamvpar(nvar) [format "%d" [string range $varline 0 [expr $space - 1]]]
    set varline [string range $varline [expr $space + 1] end]

    $ivar_table_w configure -rows [expr $gamvpar(nvar) + 1]
    for { set ii 1 } { $ii <= $gamvpar(nvar) } { incr ii } {
	### setting up the variables table
	set ivar_vals($ii,label) [iwidgets::labeledwidget $ivar_table_w.label$ii \
				      -labeltext "$ii" \
				      -background gray50 \
				      -foreground white]
	update
	
	set ivar_vals($ii,combo) [iwidgets::combobox $ivar_table_w.combo$ii \
				      -textvariable gamvpar(ivar,$ii) \
				      -validate {regexpvalidate %c} \
				      -listheight 60 \
				      -width 29]
	
	$ivar_vals($ii,combo) delete list 0
	foreach i $dim_name_data {
	    $ivar_vals($ii,combo) insert list end $i
	}
	$ivar_vals($ii,combo) delete list 0
	set space [string first " " $varline]
	set gamvpar(ivar) [format "%d" [string range $varline 0 [expr $space - 1]]]
	set varline [string range $varline [expr $space + 1] end]
	$ivar_vals($ii,combo) selection set [expr $gamvpar(ivar) - 1]
	update
	
	$ivar_table_w window configure $ii,0 -window $ivar_vals($ii,label) \
	    -relief flat -background gray50
	$ivar_table_w window configure $ii,1 -window $ivar_vals($ii,combo)
    }

    ### reads the tmin, tmax: trimming limits
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(tmin) [format "%g" [string range $line 0 [expr $space - 1]]]
    set line [string range $line [expr $space + 1] end]
    set space [string first " " $line]
    set gamvpar(tmax) [format "%g" [string range $line 0 [expr $space - 1]]]

    ### reads the outfl: file for simulation output
    set line [gets $inf]
    set backslash [string first "\\" $line]
    if { $backslash >= 0 } {
	set gamvpar(outfl) [string trimright [format "%s" [string range $line 0 [expr $backslash - 1]]]]
    } elseif { $backslash < 0 } {
	set gamvpar(outfl) $line
    }

    ### reads the nlag: number of lags
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(nlag) [format "%d" [string range $line 0 $space]]

    ### reads the xlag: unit lag separation distance
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(xlag) [format "%g" [string range $line 0 $space]]

    ### reads the xltol: lag tolerance
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(xltol) [format "%g" [string range $line 0 $space]]

    ### reads the ndir: number of variogram directions
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(ndir) [format "%d" [string range $line 0 $space]]

    ### setting up the vario_dir (variogram direction) table
    $vario_dir_table_w configure -rows [expr $gamvpar(ndir) + 1]
    for { set ii 1 } { $ii <= $gamvpar(ndir) } { incr ii } {

	set line [gets $inf]

	set vario_dir_vals($ii,label) [iwidgets::labeledwidget $vario_dir_table_w.vals$ii \
					   -labeltext "$ii" \
					   -background gray50 \
					   -foreground white]
	update
	
	set space [string first " " $line]
	set gamvpar(azm,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_azimuth($ii,entry) [iwidgets::entryfield $vario_dir_table_w.azimuth$ii \
					      -textvariable gamvpar(azm,$ii) \
					      -validate {check_number_vario_dir_azimuth %W %P %c}]
	update
	
	set space [string first " " $line]
	set gamvpar(atol,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_az_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.az_tol$ii \
					     -textvariable gamvpar(atol,$ii) \
					     -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	update
	
	set space [string first " " $line]
	set gamvpar(bandwh,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_bandwidth_h($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_h$ii \
						  -textvariable gamvpar(bandwh,$ii) \
						  -validate {check_number_vario_dir_bandwidth %W %P %c}]
	update
	
	set space [string first " " $line]
	set gamvpar(dip,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_dip($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip$ii \
					  -textvariable gamvpar(dip,$ii) \
					  -validate {check_number_vario_dir_dip %W %P %c}]
	update
	
	set space [string first " " $line]
	set gamvpar(dtol,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_dip_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip_tol$ii \
					      -textvariable gamvpar(dtol,$ii) \
					      -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	update
	
	set space [string first " " $line]
	set gamvpar(bandwd,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	set vario_dir_bandwidth_v($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_v$ii \
						  -textvariable gamvpar(bandwd,$ii) \
						  -validate {check_number_vario_dir_bandwidth %W %P %c}]
	update
	
	$vario_dir_table_w window configure $ii,0 -window $vario_dir_vals($ii,label) -relief flat -background gray50
	$vario_dir_table_w window configure $ii,1 -window $vario_dir_azimuth($ii,entry)
	$vario_dir_table_w window configure $ii,2 -window $vario_dir_az_tol($ii,entry)
	$vario_dir_table_w window configure $ii,3 -window $vario_dir_bandwidth_h($ii,entry)
	$vario_dir_table_w window configure $ii,4 -window $vario_dir_dip($ii,entry)
	$vario_dir_table_w window configure $ii,5 -window $vario_dir_dip_tol($ii,entry)
	$vario_dir_table_w window configure $ii,6 -window $vario_dir_bandwidth_v($ii,entry)
    }

    ### reads the standardize: standardization of the variograms (0=no, 1=yes)
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(standardize) [format "%d" [string range $line 0 $space]]

    ### reads the nvarg: number of variograms
    set line [gets $inf]
    set space [string first " " $line]
    set gamvpar(nvarg) [format "%d" [string range $line 0 $space]]

    ### setting up the vario_type (variogram type) table
    $vario_type_table_w configure -rows [expr $gamvpar(nvarg) + 1]
    for { set ii 1 } { $ii <= $gamvpar(nvarg) } { incr ii } {

	set line [gets $inf]

	set vario_type_vals($ii,label) [iwidgets::labeledwidget $vario_type_table_w.vals$ii \
					    -labeltext "$ii" \
					    -background gray50 \
					    -foreground white]
	update
	
	set vario_type_tail($ii,combo) [iwidgets::combobox $vario_type_table_w.tail$ii \
					    -textvariable gamvpar(ivtail,$ii) \
					    -width 15 \
					    -listheight 75 \
					    -validate {regexpvalidate %c} \
					    -selectioncommand "check_variable_tail $ii"]
	$vario_type_tail($ii,combo) clear list
	foreach i $dim_name_data {
	    $vario_type_tail($ii,combo) insert list end $i
	}
	$vario_type_tail($ii,combo) delete list 0
	set space [string first " " $line]
	set gamvpar(ivtail,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]

	set cnt 0
	foreach i $dim_name_data {
	    if { $i == [$ivar_vals($gamvpar(ivtail,$ii),combo) getcurselection] } {
		set sel $cnt
	    }
	    incr cnt
	}
	set line [string range $line [expr $space + 1] end]
	$vario_type_tail($ii,combo) selection set [expr $sel - 1]
	update

	set vario_type_head($ii,combo) [iwidgets::combobox $vario_type_table_w.head$ii \
					    -textvariable gamvpar(ivhead,$ii) \
					    -width 15 \
					    -listheight 75 \
					    -validate {regexpvalidate %c} \
					    -selectioncommand "check_variable_head $ii"]
	$vario_type_head($ii,combo) clear list
	foreach i $dim_name_data {
	    $vario_type_head($ii,combo) insert list end $i
	}
	$vario_type_head($ii,combo) delete list 0
	set space [string first " " $line]
	set gamvpar(ivhead,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]

	set cnt 0
	foreach i $dim_name_data {
	    if { $i == [$ivar_vals($gamvpar(ivhead,$ii),combo) getcurselection] } {
		set sel $cnt
	    }
	    incr cnt
	}
	set line [string range $line [expr $space + 1] end]
	$vario_type_head($ii,combo) selection set [expr $sel - 1]
	update

	set vario_type_type($ii,combo) [iwidgets::combobox $vario_type_table_w.type$ii \
					    -textvariable gamvpar(ivtype,$ii) \
					    -selectioncommand "threshold_activator $ii" \
					    -width 40 \
					    -listheight 75 \
					    -validate {regexpvalidate %c}]
	$vario_type_type($ii,combo) clear list
	$vario_type_table_w.type$ii insert list end \
	    "Traditional semi-variogram" "Traditional cross semi-variogram" "Covariance" "Correlogram" "General relative semivariogram" "Pairwise relative semi-variogram" "Semivariogram of logarithms" "Semi-madogram" "Indicator semi-variogram - continous variable" "Indicator semi-variogram - categorical variable"
	set space [string first " " $line]
	set gamvpar(ivtype,$ii) [format "%d" [string range $line 0 [expr $space - 1]]]
	set line [string range $line [expr $space + 1] end]
	$vario_type_type($ii,combo) selection set [expr $gamvpar(ivtype,$ii) - 1]
	update
	
	if { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - continous variable" || \
		 $gamvpar(ivtype,$ii) == "Indicator semi-variogram - categorical variable" } {
	    set space [string first " " $line]
	    set gamvpar(cut,$ii) [format "%g" [string range $line 0 [expr $space - 1]]]
	} else {
	    set gamvpar(cut,$ii) 0
	}

	set vario_type_threshold($ii,entry) [iwidgets::entryfield $vario_type_table_w.threshold$ii \
						 -textvariable gamvpar(cut,$ii) \
						 -validate "check_number_vario_type_threshold %W %P %c $ii"]
	update

	if { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - continous variable" || \
		 $gamvpar(ivtype,$ii) == "Indicator semi-variogram - categorical variable" } {
	    $vario_type_threshold($ii,entry) configure -state normal -textbackground white -foreground black
	} else {
	    $vario_type_threshold($ii,entry) configure -state disabled -textbackground gray85 -foreground gray50
	}

	$vario_type_table_w window configure $ii,0 \
	    -window $vario_type_vals($ii,label) -relief flat -background gray50
	$vario_type_table_w window configure $ii,1 -window $vario_type_tail($ii,combo)
	$vario_type_table_w window configure $ii,2 -window $vario_type_head($ii,combo)
	$vario_type_table_w window configure $ii,3 -window $vario_type_type($ii,combo)
	$vario_type_table_w window configure $ii,4 -window $vario_type_threshold($ii,entry)
    }

    ### reset the cancel counter to zero
    set gamvpar(cancel_counter) 0
    
    ### reset the cursor on arrow 
    . configure -cursor arrow
    
    ### close the file
    close $inf
}
#============================================================================


#############################################################################
proc write_gamv_params { fname {tmpflag 0} } {
    global tcl_platform
    global version
    global gamvpar
    global icolx_w
    global icoly_w
    global icolz_w
    global ivar_vals
    global vario_type_tail
    global vario_type_head
    global vario_type_type
    global param_error
    global gamv_box_w
    
    if { [catch {set outf [open $fname {w}]} err_msg] } {
	tk_messageBox \
	    -title "File open error" \
	    -icon error \
	    -message "The file $fname couldn't open!\n$err_msg"
    }

    ### set file name in gamv box widget
    if { $gamvpar(title) == 0 } {
	set param_dname [file dirname $fname]
	set dname_length [string length $param_dname]
	set fname_short [string range $fname [expr $dname_length + 1] end]
	wm title $gamv_box_w "GAMV parameters   -   $fname_short"
    }

    ### write header
    puts $outf "*******************************************"
    puts $outf "* File generated by GEOSSAV"
    puts $outf "* Version:       $version"
    set clockVal [clock seconds]
    puts $outf "* Date:          [clock format $clockVal -format "%A, %B %d, %Y"]"
    puts $outf "* Time:          [clock format $clockVal -format "%H:%M:%S"]"
    puts $outf "* Author:        $tcl_platform(user)"
    puts $outf "* Computer name: [info hostname]"
    puts $outf "*******************************************\n"
    
    puts $outf "Parameters for GAMV"
    puts $outf "*******************\n"
    
    puts $outf "START OF PARAMETERS:"
    
    set parameter_space 24
    set space " "
    
    ### datafl: file with data
    set comment "\\file with data"
    if { $tmpflag } {
	catch {
	    file copy -force $gamvpar(datafl) "gamv_in.dat"
	}
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "gamv_in.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $gamvpar(datafl)]
    }
    set line [format "%s" $gamvpar(datafl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### icolx, ..y, ..z: columns for x, y, z
    puts -nonewline $outf [format "%d %d " [$icolx_w curselection] [$icoly_w curselection]]
    puts -nonewline $outf [format "%d" [$icolz_w curselection]]
    set line [format "%d %d %d" [$icolx_w curselection] [$icoly_w curselection] [$icolz_w curselection]]
    set line_length [string length $line]
    set comment "\\columns for x, y, z"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### nvar, ivar(1)...ivar(nvar): number of variables and column order
    set line_length_total 0
    puts -nonewline $outf [format "%d " $gamvpar(nvar)]
    set line_length [string length $line]
    for { set ii 1 } { $ii <= $gamvpar(nvar) } { incr ii } {
	puts -nonewline $outf "[format "%d" [expr [$ivar_vals($ii,combo) curselection] + 1]] "
	set line [format "%d" [expr [$ivar_vals($ii,combo) curselection] + 1]]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
    }
    set line_length_total [expr $line_length_total + 2]
    set comment "\\number of variables and column order"
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### tmin, tmax: trimming limits
    catch {scan $gamvpar(tmin) {%g} gamvpar(tmin)}
    catch {scan $gamvpar(tmax) {%g} gamvpar(tmax)}
    puts -nonewline $outf [format "%g %g" $gamvpar(tmin) $gamvpar(tmax)]
    set line [format "%g %g" $gamvpar(tmin) $gamvpar(tmax)]
    set line_length [string length $line]
    set comment "\\trimming limits"
    puts $outf [format "%-[expr [expr $parameter_space - 0] - $line_length]s %s" $space $comment]

    ### outfl: file for output
    set comment "\\file for output"
    if { $tmpflag } {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " "gamv_out.dat"]
    } else {
	puts -nonewline $outf [format "%-[expr $parameter_space - 0]s " $gamvpar(outfl)]
    }
    set line [format "%s" $gamvpar(outfl)]
    set line_length_total [string length $line]
    if { $line_length_total < $parameter_space } {
	puts $outf [format "%s" $comment]
    } elseif { $line_length_total >= $parameter_space } {
	set parameter_space [expr $line_length_total + 1]
	puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	set parameter_space 24
    }

    ### nlag: number of lags
    set comment "\\number of lags"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $gamvpar(nlag) $comment]

    ### xlag: lag separation distance
    set comment "\\lag separation distance"
    puts $outf [format "%-[expr $parameter_space - 0]g %s" $gamvpar(xlag) $comment]

    ### xltol: lag tolerance
    set comment "\\lag tolerance"
    puts $outf [format "%-[expr $parameter_space - 0]g %s" $gamvpar(xltol) $comment]

    ### ndir: number of variogram directions
    set comment "\\number of variogram directions"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $gamvpar(ndir) $comment]

    ### azm, atol, bandwh, dip, diptol, bandwd: azimuth, azm tolerance, azm bandwidth, dip, dip tolerance, dip bandwidth 
    for { set ii 1 } { $ii <= $gamvpar(ndir) } { incr ii } {
	set line_length_total 0
	puts -nonewline $outf "[format "%g" $gamvpar(azm,$ii)] "
	puts -nonewline $outf "[format "%g" $gamvpar(atol,$ii)] "
	puts -nonewline $outf "[format "%g" $gamvpar(bandwh,$ii)] "
	puts -nonewline $outf "[format "%g" $gamvpar(dip,$ii)] "
	puts -nonewline $outf "[format "%g" $gamvpar(dtol,$ii)] "
	puts -nonewline $outf "[format "%g" $gamvpar(bandwd,$ii)] "
	set line [format "%g %g %g %g %g %g" $gamvpar(azm,$ii) $gamvpar(atol,$ii) $gamvpar(bandwh,$ii) \
		      $gamvpar(dip,$ii) $gamvpar(dtol,$ii) $gamvpar(bandwd,$ii)]
	set line_length [string length $line]
	set line_length_total [expr $line_length_total + $line_length + 1]
	set comment "\\azimuth, azm tol, azm bandh, dip, dip tol, dip bandv"
	if { $line_length_total < $parameter_space } {
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	} elseif { $line_length_total >= $parameter_space } {
	    set parameter_space [expr $line_length_total + 1]
	    puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    set parameter_space 24
	}	
    }

    ### standardize: standardize sill (0=no, 1=yes)
    set comment "\\standardize sill (0=no, 1=yes)"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $gamvpar(standardize) $comment]

    ### nvarg: number of variograms
    set comment "\\number of variograms"
    puts $outf [format "%-[expr $parameter_space - 0]d %s" $gamvpar(nvarg) $comment]

    ### ivtail, ivhead, ivtype, cut/cat: tail variable, head variable, variogram type, cutoff/category
    set param_error_tail 0
    set param_error_head 0
    for { set ii 1 } { $ii <= $gamvpar(nvarg) } { incr ii } {
	set line_length_total 0
	for { set i 1 } { $i <= $gamvpar(nvar) } { incr i } {
	    if { "[$vario_type_tail($ii,combo) getcurselection]" == "[$ivar_vals($i,combo) getcurselection]" } {
		puts -nonewline $outf "[format "%d" $i] "
		set param_error_tail [expr $param_error_tail + 0]
	    } elseif { "[$vario_type_tail($ii,combo) getcurselection]" != "[$ivar_vals($i,combo) getcurselection]" } {
		set param_error_tail [expr $param_error_tail + 1]
	    }
	}
	for { set i 1 } { $i <= $gamvpar(nvar) } { incr i } {
	    if { "[$vario_type_head($ii,combo) getcurselection]" == "[$ivar_vals($i,combo) getcurselection]" } {
		puts -nonewline $outf "[format "%d" $i] "
		set param_error_head [expr $param_error_head + 0]
	    } elseif { "[$vario_type_head($ii,combo) getcurselection]" != "[$ivar_vals($i,combo) getcurselection]" } {
		set param_error_head [expr $param_error_head + 1]
	    }
	}
	puts -nonewline $outf "[format "%d" [expr [$vario_type_type($ii,combo) curselection] + 1]] "
	if { [expr [$vario_type_type($ii,combo) curselection] + 1] >= 9 } {
	    puts -nonewline $outf "[format "%g" $gamvpar(cut,$ii)] "
	    set line [format "%d %d %d %g" [expr [$vario_type_tail($ii,combo) curselection] + 1] \
			  [expr [$vario_type_head($ii,combo) curselection] + 1] \
			  [expr [$vario_type_type($ii,combo) curselection] + 1] \
			  $gamvpar(cut,$ii)]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	    set comment "\\tail var, head var, variogram type, cutoff/category"
	    if { $line_length_total < $parameter_space } {
		puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    } elseif { $line_length_total >= $parameter_space } {
		set parameter_space [expr $line_length_total + 1]
		puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		set parameter_space 24
	    }	
	} elseif { [expr [$vario_type_type($ii,combo) curselection] + 1] < 9 } {
	    set line [format "%d %d %d" [expr [$vario_type_tail($ii,combo) curselection] + 1] \
			  [expr [$vario_type_head($ii,combo) curselection] + 1] \
			  [expr [$vario_type_type($ii,combo) curselection] + 1]]
	    set line_length [string length $line]
	    set line_length_total [expr $line_length_total + $line_length + 1]
	    set comment "\\tail var, head var, variogram type"
	    if { $line_length_total < $parameter_space } {
		puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
	    } elseif { $line_length_total >= $parameter_space } {
		set parameter_space [expr $line_length_total + 1]
		puts $outf [format "%[expr [expr $parameter_space - 0] - $line_length_total]s %s" $space $comment]
		set parameter_space 24
	    }	
	}
    }
    if { $param_error_tail > [expr ($gamvpar(nvar) * $gamvpar(nvarg)) - $gamvpar(nvarg)] || \
	     $param_error_head > [expr ($gamvpar(nvar) * $gamvpar(nvarg)) - $gamvpar(nvarg)] } {
	tk_messageBox \
	    -title "GAMV parameter file" \
	    -icon warning \
	    -message "Your settings in the parameter file are inconsistent.\nYou should correct the parameter file." \
	    -type ok
    }
    close $outf
}
#============================================================================

#
# End of proc section.
#
#############################################################################


#############################################################################
proc geossav_gamv_main { } {
    global exedir

    source $exedir/Geossav_gamv_globals.tcl
    
    . configure -cursor watch
    update
    
    set gamv_box_w [toplevel .gamv_box -class Dialog]
    wm resizable $gamv_box_w false false
    wm sizefrom $gamv_box_w program
    wm group $gamv_box_w .
    wm transient $gamv_box_w [winfo toplevel [winfo parent $gamv_box_w]]
    wm title $gamv_box_w "GAMV parameters"
    
    set geostr [wm geometry .]
    regexp {([0-9]*)x([0-9]*)(\+)([0-9]*)(\+)([0-9]*)} $geostr all width height p1 pos_x p2 pos_y
    set pos_x [expr $pos_x + $width / 2 + 212]
    set pos_y [expr $pos_y + $height / 2 + 15]
    wm geometry $gamv_box_w "+$pos_x+$pos_y"
    
    
    #############################################################################
    # Tabnotebook
    #############################################################################
    set gamv_tabnotebook_w [iwidgets::tabnotebook $gamv_box_w.gamv_tbn \
				-width 5.4i \
				-height 3.3i \
				-tabpos n \
				-tabforeground gray50 \
				-background gray85 \
				-bevelamount 2]
    
    pack $gamv_tabnotebook_w -padx 0 -pady 0
    
    
    #
    # page1: Geossav GAMV-parameters: Files
    #
    set page [$gamv_box_w.gamv_tbn add \
		  -label "Files"]
    
    
    # initialize dimension names for data
    if { ![gamv_read_data $gamvpar(datafl)] } {
	set dim_name_data {}
	lappend dim_name_data "None"
	for { set i 1} { $i <= 4 } { incr i } {
	    lappend dim_name_data $i
	}
    }	                                                      
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    
    # fields for input data
    set input_iframe_w [iwidgets::labeledframe $work_frame_w.input \
			    -labeltext "Input" \
			    -labelpos nw \
			    -borderwidth 2 \
			    -relief groove]
    set input_frame_w [$input_iframe_w childsite]; # here we have to get the actual label widget
    
    set subframe1_w [frame $input_frame_w.subfr1]
    set datafl_w [iwidgets::entryfield $subframe1_w.datafl \
		      -labeltext "File " \
		      -width 67 \
		      -state disabled \
		      -textbackground gray85 \
		      -foreground gray50 \
		      -textvariable gamvpar(datafl)]

    if { $gamvpar(datafl) != {*.dat} } {
	$datafl_w configure -state normal -textbackground white -foreground black
    }
    
    set datafl_browse_w [iwidgets::pushbutton $subframe1_w.datafl_browse \
			     -height 27 \
			     -width 40 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			     -command {
				 set tmp_fname [tk_getOpenFile \
						    -defaultextension ".dat" \
						    -filetypes {{{Geo-EAS Files} {.dat}} {{All Files} {.*}}} \
						    -initialfile $gamvpar(datafl) \
						    -initialdir $exedir \
						    -title "Select GAMV input data"]
				 
				 if { $tmp_fname != {} } {
				     
				     if { $gamvpar(datafl) == {*.dat} } {
					 for { set ii 1 } { $ii <= 1} { incr ii } {
					     destroy $ivar_vals($ii,label)
					     destroy $ivar_vals($ii,combo)
					     
					     destroy $vario_dir_vals($ii,label)
					     destroy $vario_dir_azimuth($ii,entry)
					     destroy $vario_dir_az_tol($ii,entry)
					     destroy $vario_dir_bandwidth_h($ii,entry)
					     destroy $vario_dir_dip($ii,entry)
					     destroy $vario_dir_dip_tol($ii,entry)
					     destroy $vario_dir_bandwidth_v($ii,entry)
					     
					     destroy $vario_type_vals($ii,label)
					     destroy $vario_type_tail($ii,combo)
					     destroy $vario_type_head($ii,combo)
					     destroy $vario_type_type($ii,combo)
					     destroy $vario_type_threshold($ii,entry)
					 }
					 
				     } elseif { $gamvpar(datafl) != {*.dat} } {
					 for { set ii 1 } { $ii <= $gamvpar(nvar) } { incr ii } {
					     destroy $ivar_vals($ii,label)
					     destroy $ivar_vals($ii,combo)
					 }
					 for { set ii 1 } { $ii <= $gamvpar(ndir) } { incr ii } {
					     destroy $vario_dir_vals($ii,label)
					     destroy $vario_dir_azimuth($ii,entry)
					     destroy $vario_dir_az_tol($ii,entry)
					     destroy $vario_dir_bandwidth_h($ii,entry)
					     destroy $vario_dir_dip($ii,entry)
					     destroy $vario_dir_dip_tol($ii,entry)
					     destroy $vario_dir_bandwidth_v($ii,entry)
					 }
					 for { set ii 1 } { $ii <= $gamvpar(nvarg) } { incr ii } {
					     destroy $vario_type_vals($ii,label)
					     destroy $vario_type_tail($ii,combo)
					     destroy $vario_type_head($ii,combo)
					     destroy $vario_type_type($ii,combo)
					     destroy $vario_type_threshold($ii,entry)
					 }
				     }
				 
				     ### exit from the GAMV parameters tabnotebook, indicator and counter 
				     set gamvpar(cancel_counter) 0
				 
				     ### select file name and data
				     set gamvpar(datafl) $tmp_fname
				     unset dim_name_data
				     if { [gamv_read_data $gamvpar(datafl)] != 1 } {
					 set dim_name_data {}
					 lappend dim_name_data "None"
				     }
				     
				     $icolx_w clear list
				     foreach i $dim_name_data {
					 $icolx_w insert list end $i
				     }
				     if { $gamvpar(variables) >= 4 } {
					 $icolx_w selection set 1
					 set tmp_icolx [lindex $dim_name_data 1]
				     } elseif { $gamvpar(variables) < 4 } {
					 $icolx_w selection set 1
					 set tmp_icolx [lindex $dim_name_data 1]
				     }
				     
				     $icoly_w clear list
				     foreach i $dim_name_data {
					 $icoly_w insert list end $i
				     }
				     if { $gamvpar(variables) >= 4 } {
					 $icoly_w selection set 2
					 set tmp_icoly [lindex $dim_name_data 2]
				     } elseif { $gamvpar(variables) < 4 } {
					 $icoly_w selection set 1
					 set tmp_icoly [lindex $dim_name_data 1]
				     }
				     
				     $icolz_w clear list
				     foreach i $dim_name_data {
					 $icolz_w insert list end $i
				     }
				     if { $gamvpar(variables) >= 4 } {
					 $icolz_w selection set 3
					 set tmp_icolz [lindex $dim_name_data 3]
				     } elseif { $gamvpar(variables) < 4 } {
					 $icolz_w selection set 1
					 set tmp_icolz [lindex $dim_name_data 1]
				     }

				     ### set datafl widget normal
				     $datafl_w configure -state normal -textbackground white -foreground black
				     
				     ### output path- and filename
				     if { $gamvpar(datafl) != {*.dat} } {
					 set datafl_rootname [file rootname $gamvpar(datafl)]
					 set datafl_dirname [file dirname $gamvpar(datafl)]
					 set datafl_dirname_lenght [string length $datafl_dirname]
					 set datafl_name [string range $datafl_rootname \
							      [expr $datafl_dirname_lenght + 1] end]
					 
					 ### output data path- and filename
					 set outfl_extension [file extension $gamvpar(outfl)]
					 set simulation "/GAMV_"
					 regsub {} $outfl_extension $datafl_name outputfl_1
					 regsub {} $outputfl_1 $simulation outputfl_2
					 regsub {} $outputfl_2 $datafl_dirname outputfl_3
					 set gamvpar(outfl) $outputfl_3
				     }
				     
				     ### setting up the rows of the three tables
				     $ivar_table_w configure -rows 2
				     $vario_dir_table_w configure -rows 2
				     $vario_type_table_w configure -rows 2
				     
				     for {set ii 1} {$ii < 2} {incr ii} {
					 ### setting up the variables table
					 set ivar_vals($ii,label) [iwidgets::labeledwidget $ivar_table_w.label$ii \
								       -labeltext "$ii" \
								       -background gray50 \
								       -foreground white]
					 update
					 
					 set ivar_vals($ii,combo) [iwidgets::combobox $ivar_table_w.combo$ii \
								       -textvariable gamvpar(ivar,$ii) \
								       -validate {regexpvalidate %c} \
								       -listheight 60 \
								       -width 29]
					 
					 $ivar_vals($ii,combo) clear list
					 foreach i $dim_name_data {
					     $ivar_vals($ii,combo) insert list end $i
					 }
					 $ivar_vals($ii,combo) delete list 0
					 if { $gamvpar(variables) >= 4 } {
					     $ivar_vals($ii,combo) selection set [expr $ii + 2]
					     set gamvpar(ivar) [lindex $dim_name_data [expr $ii + 3]]
					 } elseif { $gamvpar(variables) < 4 } {
					     $ivar_vals($ii,combo) selection set 0
					     set gamvpar(ivar) [lindex $dim_name_data 0]
					 }
					 update
					 
					 $ivar_table_w window configure $ii,0 -window $ivar_vals($ii,label) \
					     -relief flat -background gray50
					 $ivar_table_w window configure $ii,1 -window $ivar_vals($ii,combo)
					 
					 set gamvpar(nvar) $ii
					 
					 ### setting up the vario_dir (variogram direction) table
					 set vario_dir_vals($ii,label) [iwidgets::labeledwidget \
									    $vario_dir_table_w.vals$ii \
									    -labeltext "$ii" \
									    -background gray50 \
									    -foreground white]
					 
					 set gamvpar(azm,$ii) 0
					 set vario_dir_azimuth($ii,entry) \
					     [iwidgets::entryfield $vario_dir_table_w.azimuth$ii \
						  -textvariable gamvpar(azm,$ii) \
						  -validate {check_number_vario_dir_azimuth %W %P %c}]
					 
					 set gamvpar(atol,$ii) 90
					 set vario_dir_az_tol($ii,entry) \
					     [iwidgets::entryfield $vario_dir_table_w.az_tol$ii \
						  -textvariable gamvpar(atol,$ii) \
						  -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
					 
					 set gamvpar(bandwh,$ii) 100
					 set vario_dir_bandwidth_h($ii,entry) \
					     [iwidgets::entryfield $vario_dir_table_w.bandwidth_h$ii \
						  -textvariable gamvpar(bandwh,$ii) \
						  -validate {check_number_vario_dir_bandwidth %W %P %c}]
					 
					 set gamvpar(dip,$ii) 0
					 set vario_dir_dip($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip$ii \
									   -textvariable gamvpar(dip,$ii) \
									   -validate {check_number_vario_dir_dip %W %P %c}]
					 
					 set gamvpar(dtol,$ii) 90
					 set vario_dir_dip_tol($ii,entry) \
					     [iwidgets::entryfield $vario_dir_table_w.dip_tol$ii \
						  -textvariable gamvpar(dtol,$ii) \
						  -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
					 
					 set gamvpar(bandwd,$ii) 100
					 set vario_dir_bandwidth_v($ii,entry) \
					     [iwidgets::entryfield $vario_dir_table_w.bandwidth_v$ii \
						  -textvariable gamvpar(bandwd,$ii) \
						  -validate {check_number_vario_dir_bandwidth %W %P %c}]
					 
					 $vario_dir_table_w window configure $ii,0 \
					     -window $vario_dir_vals($ii,label) -relief flat -background gray50
					 $vario_dir_table_w window configure $ii,1 -window $vario_dir_azimuth($ii,entry)
					 $vario_dir_table_w window configure $ii,2 -window $vario_dir_az_tol($ii,entry)
					 $vario_dir_table_w window configure $ii,3 \
					     -window $vario_dir_bandwidth_h($ii,entry)
					 $vario_dir_table_w window configure $ii,4 -window $vario_dir_dip($ii,entry)
					 $vario_dir_table_w window configure $ii,5 -window $vario_dir_dip_tol($ii,entry)
					 $vario_dir_table_w window configure $ii,6 \
					     -window $vario_dir_bandwidth_v($ii,entry)
					 
					 set gamvpar(ndir) $ii
					 
					 ### setting up the vario_type (variogram type) table
					 set vario_type_vals($ii,label) \
					     [iwidgets::labeledwidget $vario_type_table_w.vals$ii \
						  -labeltext "$ii" \
						  -background gray50 \
						  -foreground white]
					 
					 set gamvpar(ivtail,$ii) 1
					 set vario_type_tail($ii,combo) [iwidgets::combobox $vario_type_table_w.tail$ii \
									     -textvariable gamvpar(ivtail,$ii) \
									     -width 15 \
									     -listheight 75 \
									     -validate {regexpvalidate %c} \
									     -selectioncommand "check_variable_tail $ii"]
					 foreach i $dim_name_data {
					     $vario_type_tail($ii,combo) insert list end $i
					 }
					 $vario_type_tail($ii,combo) delete list 0
					 if { $gamvpar(variables) >= 4 } {
					     $vario_type_tail($ii,combo) selection set 3
					     set gamvpar(ivtail,$ii) [lindex $dim_name_data 4]
					 } elseif { $gamvpar(variables) < 4 } {
					     $vario_type_tail($ii,combo) selection set 0
					     set gamvpar(ivtail,$ii) [lindex $dim_name_data 1]
					 }
					 
					 set gamvpar(ivhead,$ii) 1
					 set vario_type_head($ii,combo) [iwidgets::combobox $vario_type_table_w.head$ii \
									     -textvariable gamvpar(ivhead,$ii) \
									     -width 15 \
									     -listheight 75 \
									     -validate {regexpvalidate %c} \
									     -selectioncommand "check_variable_head $ii"]
					 foreach i $dim_name_data {
					     $vario_type_head($ii,combo) insert list end $i
					 }
					 $vario_type_head($ii,combo) delete list 0
					 if { $gamvpar(variables) >= 4 } {
					     $vario_type_head($ii,combo) selection set 3
					     set gamvpar(ivhead,$ii) [lindex $dim_name_data 4]
					 } elseif { $gamvpar(variables) < 4 } {
					     $vario_type_head($ii,combo) selection set 0
					     set gamvpar(ivhead,$ii) [lindex $dim_name_data 1]
					 }
					 
					 set gamvpar(ivtype,$ii) 0
					 set vario_type_type($ii,combo) [iwidgets::combobox $vario_type_table_w.type$ii \
									     -textvariable gamvpar(ivtype,$ii) \
									     -selectioncommand "threshold_activator $ii" \
									     -width 40 \
									     -listheight 75 \
									     -validate {regexpvalidate %c}]
					 set sel 0   ;# that's the default selection if gamvpar(ivtype,$ii) is empty
					 $vario_type_table_w.type$ii insert list end \
					     "Traditional semi-variogram" "Traditional cross semi-variogram" "Covariance" "Correlogram" "General relative semivariogram" "Pairwise relative semi-variogram" "Semivariogram of logarithms" "Semi-madogram" "Indicator semi-variogram - continous variable" "Indicator semi-variogram - categorical variable"
					 if { $gamvpar(ivtype,$ii) == "Traditional semi-variogram" } {
					     set sel 0	
					 } elseif { $gamvpar(ivtype,$ii) == "Traditional cross semi-variogram" } {
					     set sel 1
					 } elseif { $gamvpar(ivtype,$ii) == "Covariance" } {
					     set sel 2
					 } elseif { $gamvpar(ivtype,$ii) == "Correlogram" } {
					     set sel 3
					 } elseif { $gamvpar(ivtype,$ii) == "General relative semivariogram" } {
					     set sel 4
					 } elseif { $gamvpar(ivtype,$ii) == "Pairwise relative semi-variogram" } {
					     set sel 5
					 } elseif { $gamvpar(ivtype,$ii) == "Semivariogram of logarithms" } {
					     set sel 6
					 } elseif { $gamvpar(ivtype,$ii) == "Semi-madogram" } {
					     set sel 7
					 } elseif {$gamvpar(ivtype,$ii) == \
						       "Indicator semi-variogram - continous variable"} {
					     set sel 8
					 } elseif {$gamvpar(ivtype,$ii) == \
						       "Indicator semi-variogram - categorical variable"} {
					     set sel 9
					 }
					 $vario_type_type($ii,combo) selection set $sel
					 set $vario_type_type($ii,combo) [lindex $vario_type_table_w.type$ii $sel]
					 
					 set gamvpar(cut,$ii) 0
					 set vario_type_threshold($ii,entry) \
					     [iwidgets::entryfield $vario_type_table_w.threshold$ii \
						  -textvariable gamvpar(cut,$ii) \
						  -state disabled \
						  -textbackground gray85 \
						  -foreground gray50 \
						  -validate "check_number_vario_type_threshold %W %P %c $ii"]
					 
					 $vario_type_table_w window configure $ii,0 \
					     -window $vario_type_vals($ii,label) -relief flat -background gray50
					 $vario_type_table_w window configure $ii,1 -window $vario_type_tail($ii,combo)
					 $vario_type_table_w window configure $ii,2 -window $vario_type_head($ii,combo)
					 $vario_type_table_w window configure $ii,3 -window $vario_type_type($ii,combo)
					 $vario_type_table_w window configure $ii,4 \
					     -window $vario_type_threshold($ii,entry)
					 
					 set gamvpar(nvarg) $ii
				     }
				 }
			     }]
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $datafl_w        -row 0 -column 0 -sticky we -pady 0
    grid $datafl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    set subframe2_w [frame $input_frame_w.subfr2]
    
    if { $gamvpar(datafl) != {*.dat} } {
	set gamvpar(cancel_counter) [expr $gamvpar(cancel_counter) + 1]
    }
    
    set icolx_w [iwidgets::combobox $subframe2_w.icolx \
		     -labeltext "X " \
		     -textvariable gamvpar(icolx) \
		     -width 17 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $gamvpar(datafl) == {*.dat} } {
	foreach i $dim_name_data {
	    $icolx_w insert list end $i
	}
	$icolx_w selection set 1
	set gamvpar(icolx) [lindex $dim_name_data 1]
    } elseif { $gamvpar(datafl) != {*.dat} } {
	set tmp_icolx $gamvpar(icolx)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if gamvpar(icolx) is empty
	foreach i $dim_name_data {
	    $icolx_w insert list end $i
	    $icolx_w delete list [expr $gamvpar(variables) + 1]
	    if { $i == $tmp_icolx } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	$icolx_w selection set $sel
	set gamvpar(icolx) [lindex $dim_name_data $sel]
    }
    
    
    set icoly_w [iwidgets::combobox $subframe2_w.icoly \
		     -labeltext "Y " \
		     -textvariable gamvpar(icoly) \
		     -width 17 \
		     -listheight 74 \
		 -validate {regexpvalidate %c}]
    if { $gamvpar(datafl) == {*.dat} } {
	foreach i $dim_name_data {
	    $icoly_w insert list end $i
	}
	$icoly_w selection set 2
	set gamvpar(icoly) [lindex $dim_name_data 2]
    } elseif { $gamvpar(datafl) != {*.dat} } {
	set tmp_icoly $gamvpar(icoly)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if gamvpar(icoly) is empty
	foreach i $dim_name_data {
	    $icoly_w insert list end $i
	    $icoly_w delete list [expr $gamvpar(variables) + 1]
	    if { $i == $tmp_icoly } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	$icoly_w selection set $sel
	set gamvpar(icoly) [lindex $dim_name_data $sel]
    }


    set icolz_w [iwidgets::combobox $subframe2_w.icolz \
		     -labeltext "Z " \
		     -textvariable gamvpar(icolz) \
		     -width 17 \
		     -listheight 74 \
		     -validate {regexpvalidate %c}]
    if { $gamvpar(datafl) == {*.dat} } {
	foreach i $dim_name_data {
	    $icolz_w insert list end $i
	}
	$icolz_w selection set 3
	set gamvpar(icolz) [lindex $dim_name_data 3]
    } elseif { $gamvpar(datafl) != {*.dat} } {
	set tmp_icolz $gamvpar(icolz)
	set cnt 0   ;# counter starts at 0
	set sel 1   ;# that's the default selection if gamvpar(icolz) is empty
	foreach i $dim_name_data {
	    $icolz_w insert list end $i
	    $icolz_w delete list [expr $gamvpar(variables) + 1]
	    if { $i == $tmp_icolz } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		set sel $cnt
	    }
	    incr cnt
	}
	set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	$icolz_w selection set $sel
	set gamvpar(icolz) [lindex $dim_name_data $sel]
    }
    
    
    grid columnconfigure $subframe2_w 0 -minsize 154
    grid columnconfigure $subframe2_w 1 -minsize 160
    grid columnconfigure $subframe2_w 2 -minsize 160
    
    grid $icolx_w -row 0 -column 0 -sticky e -pady 0
    grid $icoly_w -row 0 -column 1 -sticky e -pady 0
    grid $icolz_w -row 0 -column 2 -sticky e -pady 0
    
    
    set subframe3_w [frame $input_frame_w.subfr3]

    set plus_variables_pushbutton_w [iwidgets::pushbutton $subframe3_w.plus_pbutton \
					 -height 27 \
					 -width 27 \
					 -defaultringpad 0 \
					 -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
					 -command {plus_variables}]
    
    
    set minus_variables_pushbutton_w [iwidgets::pushbutton $subframe3_w.minus_pbutton \
					  -height 27 \
					  -width 27 \
					  -defaultringpad 0 \
					  -image [image create photo -file [file join "$exedir/images/minus.gif"]] \
					  -command {minus_variables}]
    
    
    set trimlim_min_w [iwidgets::entryfield $subframe3_w.min \
			   -labeltext "Trimming limits " \
			   -width 16 \
			   -textvariable gamvpar(tmin) \
			   -validate {check_number_t %W %P}]

    
    set trimlim_max_w [iwidgets::entryfield $subframe3_w.max \
			   -labeltext "     " \
			   -width 16 \
			   -textvariable gamvpar(tmax) \
			   -validate {check_number_t %W %P}]
    
    
    grid columnconfigure $subframe3_w 0 -minsize 10
    grid columnconfigure $subframe3_w 1 -minsize 127
    grid columnconfigure $subframe3_w 2 -minsize 195
    grid columnconfigure $subframe3_w 3 -minsize 60
    
    grid $plus_variables_pushbutton_w  -row 0 -column 0 -sticky w -pady 0
    grid $minus_variables_pushbutton_w -row 0 -column 1 -sticky w -pady 0
    grid $trimlim_min_w -row 0 -column 2 -sticky e -pady 6
    grid $trimlim_max_w -row 0 -column 3 -sticky w -pady 6
    
    
    set subframe4_w [frame $input_frame_w.subfr4]
    set ivar_frame_w [frame $subframe4_w.ivar_frame \
			  -background grey40 \
			  -borderwidth 2 \
			  -relief sunken]
    
    set ivar_sframe_w [frame $ivar_frame_w.ivar_sframe \
			   -height 90 \
			   -width 230]
    set ivar_hsbar_w [scrollbar $ivar_sframe_w.yscroll \
			  -orient vertical \
			  -command "$ivar_sframe_w.table yview"]

    if { $gamvpar(datafl) == "*.dat" } {

	set ivar_table_w [table $ivar_sframe_w.table \
			      -variable ivar_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows 2 \
			      -yscrollcommand "$ivar_hsbar_w set" \
			      -height 5]
	
	$ivar_table_w width 0 5
	$ivar_table_w width 1 29
	
	set ivar_vals(0,0) "\#"
	set ivar_vals(0,1) "Variable"
	
	for { set ii 1 } { $ii < 2 } { incr ii } {
	    
	    set gamvpar(nvar) $ii
	    
	    set ivar_vals($ii,label) [iwidgets::labeledwidget $ivar_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set ivar_vals($ii,combo) [iwidgets::combobox $ivar_table_w.combo$ii \
					  -textvariable gamvpar(ivar,$ii) \
					  -validate {regexpvalidate %c} \
					  -listheight 60 \
					  -width 29]
	    
	    foreach i $dim_name_data {
		$ivar_vals($ii,combo) insert list end $i
	    }
	    $ivar_vals($ii,combo) delete list 0
	    $ivar_vals($ii,combo) selection set [expr $ii + 2]
	    set gamvpar(ivar) [lindex $dim_name_data [expr $ii + 3]]
	    
	    
	    $ivar_table_w window configure $ii,0 -window $ivar_vals($ii,label) -relief flat -background gray50
	    $ivar_table_w window configure $ii,1 -window $ivar_vals($ii,combo)
	}

    } elseif { $gamvpar(datafl) != "*.dat" } {
    
    	set ivar_table_w [table $ivar_sframe_w.table \
			      -variable ivar_vals \
			      -titlerows 1 \
			      -titlecols 1 \
			      -cols 2 \
			      -rows [expr $gamvpar(nvar) + 1] \
			      -yscrollcommand "$ivar_hsbar_w set" \
			      -height 5]
	
	$ivar_table_w width 0 5
	$ivar_table_w width 1 29
	
	set ivar_vals(0,0) "\#"
	set ivar_vals(0,1) "Variable"
	
	for { set ii 1 } { $ii < [expr $gamvpar(nvar) + 1] } { incr ii  } {
	    
	    set ivar_vals($ii,label) [iwidgets::labeledwidget $ivar_table_w.label$ii \
					  -labeltext "$ii" \
					  -background gray50 \
					  -foreground white]
	    
	    set ivar_vals($ii,combo) [iwidgets::combobox $ivar_table_w.combo$ii \
					  -textvariable gamvpar(ivar,$ii) \
					  -validate {regexpvalidate %c} \
					  -listheight 60 \
					  -width 29]	    
	    set tmp_ivar($ii) $gamvpar(ivar,$ii)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if gamvpar(ivar) is empty
	    $ivar_vals($ii,combo) delete list 0

	    foreach i $dim_name_data {
		$ivar_vals($ii,combo) insert list end $i
		$ivar_vals($ii,combo) delete list [expr $gamvpar(variables) + 1]
		if { $i == $tmp_ivar($ii) } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    $ivar_vals($ii,combo) delete list 0
	    set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	    $ivar_vals($ii,combo) selection set [expr $sel - 1]
	    set gamvpar(ivar,$ii) [lindex $dim_name_data $sel]

	    
	    $ivar_table_w window configure $ii,0 -window $ivar_vals($ii,label) -relief flat -background gray50
	    $ivar_table_w window configure $ii,1 -window $ivar_vals($ii,combo)
	}
    }


    pack $ivar_hsbar_w  -side right -anchor w -fill y
    pack $ivar_table_w  -side left -anchor w -padx 2 -expand 1 -fill both
    pack $ivar_sframe_w -anchor nw -expand 0
    pack propagate $ivar_sframe_w 0
    
    
    set lags_iframe_w [iwidgets::labeledframe $subframe4_w.lags \
			   -labeltext "Lags" \
			   -labelpos nw \
			   -borderwidth 2 \
			   -relief groove]
    set lags_frame_w [$lags_iframe_w childsite]
    
    set tmp_nlag $gamvpar(nlag)
    set number_w [iwidgets::spinint $lags_frame_w.number_spinint \
		      -labeltext "Number of lags " \
		      -range {1 100} \
		      -width 5 \
		      -textvariable gamvpar(nlag) \
		      -validate {check_number_nlag %W %P %c}]
    set gamvpar(nlag) $tmp_nlag
    
    
    set separation_w [iwidgets::entryfield $lags_frame_w.separation_entry \
			  -labeltext "Unit lag separation distance " \
			  -width 7 \
			  -textvariable gamvpar(xlag) \
			  -validate {check_separation_tolerance %W %P %c}]
    
    set tolerance_w [iwidgets::entryfield $lags_frame_w.tolerance_entry \
			 -labeltext "Lag tolerance " \
			 -width 7 \
			 -textvariable gamvpar(xltol) \
			 -validate {check_separation_tolerance %W %P %c}]
    
    
    grid columnconfigure $lags_frame_w 0 -minsize 0
    grid columnconfigure $lags_frame_w 1 -minsize 0 
    grid columnconfigure $lags_frame_w 2 -minsize 0
    
    grid $number_w -row 0 -column 0 -sticky e -pady 2
    grid $separation_w -row 1 -column 0 -sticky e -pady 0
    grid $tolerance_w -row 2 -column 0 -sticky e -pady 2
    
    
    grid columnconfigure $subframe4_w 0 -minsize 235
    grid columnconfigure $subframe4_w 1 -minsize 250
    
    grid $ivar_frame_w -row 0 -column 0 -sticky new -rowspan 4 -pady 0
    grid $lags_iframe_w -row 0 -column 1 -sticky new -rowspan 4 -pady 0
    
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$subframe3_w \
	$subframe4_w \
	-anchor w -expand 1 -padx 3 -pady 1
    
    
    # fields for output data
    set output_iframe_w [iwidgets::labeledframe $work_frame_w.output \
			     -labeltext "Output" \
			     -labelpos nw \
			     -borderwidth 2 \
			     -relief groove]
    set output_frame_w [$output_iframe_w childsite]
    
    set subframe1_w [frame $output_frame_w.subfr1]
    set outfl_w [iwidgets::entryfield $subframe1_w.outfl_entry \
		     -labeltext "File " \
		     -width 69 \
		     -textvariable gamvpar(outfl)]
    set outfl_browse_w [iwidgets::pushbutton $subframe1_w.outfl_browse \
			    -height 27 \
			    -width 27 \
			    -defaultringpad 0 \
			    -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			    -command {
				set tmp_fname [tk_getSaveFile \
						   -defaultextension ".out" \
						   -filetypes {{{Output Files} {.out}} {{All Files} {.*}}} \
						   -initialfile $gamvpar(outfl) \
						   -initialdir $exedir \
						   -title "Save output file as ..."]

				if { $tmp_fname != {} } {
				    set gamvpar(outfl) $tmp_fname
				}
			    }]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 395
    grid columnconfigure $subframe1_w 1 -minsize 25
    
    grid $outfl_w        -row 0 -column 0 -sticky we -pady 0
    grid $outfl_browse_w -row 0 -column 1 -sticky w -pady 0
    
    
    pack \
	$subframe1_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    
    pack \
	$input_iframe_w \
	$output_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x

    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    #
    # page2: Geossav GAMV-parameters: Variograms
    #
    set page [$gamv_box_w.gamv_tbn add \
		  -label "Variograms"]
    
    
    set work_frame_w [frame $page.workframe \
			  -borderwidth 2 \
			  -relief flat]
    
    
    # fields for variograms
    set variograms_iframe_w [iwidgets::labeledframe $work_frame_w.output \
				 -labeltext "Variograms" \
				 -labelpos nw \
				 -borderwidth 2 \
				 -relief groove]
    set variograms_frame_w [$variograms_iframe_w childsite]
    
    set subframe1_w [frame $variograms_frame_w.subfr1]
    set plus_var_dir_pushbutton_w [iwidgets::pushbutton $subframe1_w.plus_var_dir_pbutton \
				       -height 27 \
				       -width 27 \
				       -defaultringpad 0 \
				       -image [image create photo -file [file join "$exedir/images/plus.gif"]] \
				       -command {plus_var_dir}]
    
    
    set minus_var_dir_pushbutton_w [iwidgets::pushbutton $subframe1_w.minus_var_pbutton \
					-height 27 \
					-width 27 \
					-defaultringpad 0 \
					-image [image create photo -file [file join "$exedir/images/minus.gif"]] \
					-command {minus_var_dir}]
    
    
    set standardize_sill_checkbutton_w [checkbutton $subframe1_w.standardize_sill_checkbutton \
					    -text "Standardize sill" \
					    -variable gamvpar(standardize)]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 10
    grid columnconfigure $subframe1_w 1 -minsize 127
    grid columnconfigure $subframe1_w 2 -minsize 150
    
    grid $plus_var_dir_pushbutton_w  -row 0 -column 0 -sticky w -pady 0
    grid $minus_var_dir_pushbutton_w -row 0 -column 1 -sticky w -pady 0
    grid $standardize_sill_checkbutton_w -row 0 -column 2 -sticky e -pady 6
    
    
    set subframe2_w [frame $variograms_frame_w.subfr2]
    set vario_dir_frame_w [frame $subframe2_w.vario_dir_frame \
			       -background grey40 \
			       -borderwidth 2 \
			       -relief sunken]
    
    set vario_dir_sframe_w [frame $vario_dir_frame_w.vario_dir_sframe \
				-height 72 \
				-width 480]
    
    set vario_dir_hsbar_w [scrollbar $vario_dir_sframe_w.yscroll \
			       -orient vertical \
			       -command "$vario_dir_sframe_w.table yview"]

    if { $gamvpar(datafl) == "*.dat" } {    
    
	### setting up the vario_dir (variogram direction) table
	set vario_dir_table_w [table $vario_dir_sframe_w.table \
				   -variable vario_dir_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 7 \
				   -rows 2 \
				   -yscrollcommand "$vario_dir_hsbar_w set" \
				   -height 4]
	
	$vario_dir_table_w width 0 4
	$vario_dir_table_w width 1 10
	$vario_dir_table_w width 2 10
	$vario_dir_table_w width 3 15
	$vario_dir_table_w width 4 10
	$vario_dir_table_w width 5 10
	$vario_dir_table_w width 6 15
	
	set vario_dir_vals(0,0) "\#"
	set vario_dir_vals(0,1) "Azimuth"
	set vario_dir_vals(0,2) "Az tol"
	set vario_dir_vals(0,3) "Bandwidth h"
	set vario_dir_vals(0,4) "Dip"
	set vario_dir_vals(0,5) "Dip tol"
	set vario_dir_vals(0,6) "Bandwidth v"
	
	
	for { set ii 1 } { $ii < 2 } { incr ii } {
	    set vario_dir_vals($ii,label) [iwidgets::labeledwidget $vario_dir_table_w.vals$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]				       
	    
	    set gamvpar(azm,$ii) 0
	    set vario_dir_azimuth($ii,entry) [iwidgets::entryfield $vario_dir_table_w.azimuth$ii \
						  -textvariable gamvpar(azm,$ii) \
						  -validate {check_number_vario_dir_azimuth %W %P %c}]
	    
	    
	    set gamvpar(atol,$ii) 90
	    set vario_dir_az_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.az_tol$ii \
						 -textvariable gamvpar(atol,$ii) \
						 -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    
	    
	    set gamvpar(bandwh,$ii) 100
	    set vario_dir_bandwidth_h($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_h$ii \
						      -textvariable gamvpar(bandwh,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    
	    
	    set gamvpar(dip,$ii) 0
	    set vario_dir_dip($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip$ii \
					      -textvariable gamvpar(dip,$ii) \
					      -validate {check_number_vario_dir_dip %W %P %c}]
	    
	    set gamvpar(dtol,$ii) 90
	    set vario_dir_dip_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip_tol$ii \
						  -textvariable gamvpar(dtol,$ii) \
						  -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    
	    set gamvpar(bandwd,$ii) 100
	    set vario_dir_bandwidth_v($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_v$ii \
						      -textvariable gamvpar(bandwd,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    
	    $vario_dir_table_w window configure $ii,0 -window $vario_dir_vals($ii,label) -relief flat -background gray50
	    $vario_dir_table_w window configure $ii,1 -window $vario_dir_azimuth($ii,entry)
	    $vario_dir_table_w window configure $ii,2 -window $vario_dir_az_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,3 -window $vario_dir_bandwidth_h($ii,entry)
	    $vario_dir_table_w window configure $ii,4 -window $vario_dir_dip($ii,entry)
	    $vario_dir_table_w window configure $ii,5 -window $vario_dir_dip_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,6 -window $vario_dir_bandwidth_v($ii,entry)
	}
    
    } elseif { $gamvpar(datafl) != "*.dat" } {

	### setting up the vario_dir (variogram direction) table
	set vario_dir_table_w [table $vario_dir_sframe_w.table \
				   -variable vario_dir_vals \
				   -titlerows 1 \
				   -titlecols 1 \
				   -cols 7 \
				   -rows [expr $gamvpar(ndir) + 1] \
				   -yscrollcommand "$vario_dir_hsbar_w set" \
				   -height 4]
	
	$vario_dir_table_w width 0 4
	$vario_dir_table_w width 1 10
	$vario_dir_table_w width 2 10
	$vario_dir_table_w width 3 15
	$vario_dir_table_w width 4 10
	$vario_dir_table_w width 5 10
	$vario_dir_table_w width 6 15
	
	set vario_dir_vals(0,0) "\#"
	set vario_dir_vals(0,1) "Azimuth"
	set vario_dir_vals(0,2) "Az tol"
	set vario_dir_vals(0,3) "Bandwidth h"
	set vario_dir_vals(0,4) "Dip"
	set vario_dir_vals(0,5) "Dip tol"
	set vario_dir_vals(0,6) "Bandwidth v"
	
	
	for { set ii 1 } { $ii < [expr $gamvpar(ndir) + 1] } { incr ii } {
	    set vario_dir_vals($ii,label) [iwidgets::labeledwidget $vario_dir_table_w.vals$ii \
					       -labeltext "$ii" \
					       -background gray50 \
					       -foreground white]				       
	    
	    set vario_dir_azimuth($ii,entry) [iwidgets::entryfield $vario_dir_table_w.azimuth$ii \
						  -textvariable gamvpar(azm,$ii) \
						  -validate {check_number_vario_dir_azimuth %W %P %c}]
	    
	    
	    set vario_dir_az_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.az_tol$ii \
						 -textvariable gamvpar(atol,$ii) \
						 -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    
	    
	    set vario_dir_bandwidth_h($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_h$ii \
						      -textvariable gamvpar(bandwh,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    
	    
	    set vario_dir_dip($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip$ii \
					      -textvariable gamvpar(dip,$ii) \
					      -validate {check_number_vario_dir_dip %W %P %c}]
	    
	    set vario_dir_dip_tol($ii,entry) [iwidgets::entryfield $vario_dir_table_w.dip_tol$ii \
						  -textvariable gamvpar(dtol,$ii) \
						  -validate {check_number_vario_dir_az_and_dip_tol %W %P %c}]
	    
	    set vario_dir_bandwidth_v($ii,entry) [iwidgets::entryfield $vario_dir_table_w.bandwidth_v$ii \
						      -textvariable gamvpar(bandwd,$ii) \
						      -validate {check_number_vario_dir_bandwidth %W %P %c}]
	    
	    $vario_dir_table_w window configure $ii,0 -window $vario_dir_vals($ii,label) -relief flat -background gray50
	    $vario_dir_table_w window configure $ii,1 -window $vario_dir_azimuth($ii,entry)
	    $vario_dir_table_w window configure $ii,2 -window $vario_dir_az_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,3 -window $vario_dir_bandwidth_h($ii,entry)
	    $vario_dir_table_w window configure $ii,4 -window $vario_dir_dip($ii,entry)
	    $vario_dir_table_w window configure $ii,5 -window $vario_dir_dip_tol($ii,entry)
	    $vario_dir_table_w window configure $ii,6 -window $vario_dir_bandwidth_v($ii,entry)
	}
    }

    
    pack $vario_dir_hsbar_w -side right -anchor w -fill y
    pack $vario_dir_table_w -side left -anchor w -padx 2 -expand 1 -fill both
    pack $vario_dir_sframe_w -anchor nw -expand 1
    pack propagate $vario_dir_sframe_w 0
    grid $vario_dir_frame_w -row 0 -column 0 -sticky news -pady 0
    
    
    set spaceframe1_w [frame $variograms_frame_w.spaceframe1 \
			   -height 10]
    
    
    set subframe3_w [frame $variograms_frame_w.subfr3]
    set plus_var_type_pushbutton_w [iwidgets::pushbutton $subframe3_w.plus_var_type_pbutton \
					-height 27 \
					-width 27 \
					-defaultringpad 0 \
					-image [image create photo -file [file join "$exedir/images/plus.gif"]] \
					-command {plus_var_type}]
    
    
    set minus_var_type_pushbutton_w [iwidgets::pushbutton $subframe3_w.minus_var_type_pbutton \
					 -height 27 \
					 -width 27 \
					 -defaultringpad 0 \
					 -image [image create photo -file [file join "$exedir/images/minus.gif"]] \
					 -command {minus_var_type}]
    
    
    grid columnconfigure $subframe1_w 0 -minsize 10
    grid columnconfigure $subframe1_w 1 -minsize 152
    
    grid $plus_var_type_pushbutton_w  -row 0 -column 0 -sticky w -pady 0
    grid $minus_var_type_pushbutton_w -row 0 -column 1 -sticky w -pady 0
    
    
    set spaceframe2_w [frame $variograms_frame_w.spaceframe2 \
			   -height 6]
    
    
    set subframe4_w [frame $variograms_frame_w.subfr4]
    set vario_type_frame_w [frame $subframe4_w.vario_type_frame \
				-background grey40 \
				-borderwidth 2 \
				-relief sunken]
    
    set vario_type_sframe_w [frame $vario_type_frame_w.vario_type_sframe \
				 -height 72 \
				 -width 480]
    
    set vario_type_hsbar_w [scrollbar $vario_type_sframe_w.yscroll \
				-orient vertical \
				-command "$vario_type_sframe_w.table yview"]
    
    if { $gamvpar(datafl) == "*.dat" } {

	### setting up the vario_type (variogram type) table
	set vario_type_table_w [table $vario_type_sframe_w.table \
				    -variable vario_type_vals \
				    -titlerows 1 \
				    -titlecols 1 \
				    -cols 5 \
				    -rows 2 \
				    -yscrollcommand "$vario_type_hsbar_w set" \
				    -height 4]
	
	$vario_type_table_w width 0 4
	$vario_type_table_w width 1 10
	$vario_type_table_w width 2 10
	$vario_type_table_w width 3 40
	$vario_type_table_w width 4 10
	
	set vario_type_vals(0,0) "\#"
	set vario_type_vals(0,1) "Tail"
	set vario_type_vals(0,2) "Head"
	set vario_type_vals(0,3) "Type"
	set vario_type_vals(0,4) "Threshold"
	
	for { set ii 1 } { $ii < 2 } { incr ii } {
	    set vario_type_vals($ii,label) [iwidgets::labeledwidget $vario_type_table_w.vals$ii \
						-labeltext "$ii" \
						-background gray50 \
						-foreground white]				       
	    
	    set gamvpar(ivtail,$ii) 1
	    set vario_type_tail($ii,combo) [iwidgets::combobox $vario_type_table_w.tail$ii \
						-textvariable gamvpar(ivtail,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_tail $ii "]
	    foreach i $dim_name_data {
		$vario_type_tail($ii,combo) insert list end $i
	    }
	    $vario_type_tail($ii,combo) delete list 0
	    $vario_type_tail($ii,combo) selection set [expr $ii + 2]
	    set gamvpar(ivtail,$ii) [lindex $dim_name_data [expr $ii + 3]]
	    
	    
	    set gamvpar(ivhead,$ii) 1
	    set vario_type_head($ii,combo) [iwidgets::combobox $vario_type_table_w.head$ii \
						-textvariable gamvpar(ivhead,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_head $ii"]
	    foreach i $dim_name_data {
		$vario_type_head($ii,combo) insert list end $i
	    }
	    $vario_type_head($ii,combo) delete list 0
	    $vario_type_head($ii,combo) selection set [expr $ii + 2]
	    set gamvpar(ivhead,$ii) [lindex $dim_name_data [expr $ii + 3]]
	    
	    
	    set gamvpar(ivtype,$ii) 0
	    set vario_type_type($ii,combo) [iwidgets::combobox $vario_type_table_w.type$ii \
						-textvariable gamvpar(ivtype,$ii) \
						-selectioncommand "threshold_activator $ii" \
						-width 40 \
						-listheight 75 \
						-validate {regexpvalidate %c}]
	    set sel 0   ;# that's the default selection if gamvpar(ivtype,$ii) is empty
	    $vario_type_table_w.type$ii insert list end \
		"Traditional semi-variogram" "Traditional cross semi-variogram" "Covariance" "Correlogram" "General relative semivariogram" "Pairwise relative semi-variogram" "Semivariogram of logarithms" "Semi-madogram" "Indicator semi-variogram - continous variable" "Indicator semi-variogram - categorical variable"
	    if { $gamvpar(ivtype,$ii) == "Traditional semi-variogram" } {
		set sel 0	
	    } elseif { $gamvpar(ivtype,$ii) == "Traditional cross semi-variogram" } {
		set sel 1
	    } elseif { $gamvpar(ivtype,$ii) == "Covariance" } {
		set sel 2
	    } elseif { $gamvpar(ivtype,$ii) == "Correlogram" } {
		set sel 3
	    } elseif { $gamvpar(ivtype,$ii) == "General relative semivariogram" } {
		set sel 4
	    } elseif { $gamvpar(ivtype,$ii) == "Pairwise relative semi-variogram" } {
		set sel 5
	    } elseif { $gamvpar(ivtype,$ii) == "Semivariogram of logarithms" } {
		set sel 6
	    } elseif { $gamvpar(ivtype,$ii) == "Semi-madogram" } {
		set sel 7
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - continous variable" } {
		set sel 8
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - categorical variable" } {
		set sel 9
	    }
	    $vario_type_type($ii,combo) selection set $sel
	    set $vario_type_type($ii,combo) [lindex $vario_type_table_w.type$ii $sel]
	    
	    
	    set gamvpar(cut,$ii) 0
	    set vario_type_threshold($ii,entry) [iwidgets::entryfield $vario_type_table_w.threshold$ii \
						     -textvariable gamvpar(cut,$ii) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate "check_number_vario_type_threshold %W %P %c $ii"]
	    
	    
	    $vario_type_table_w window configure $ii,0 -window $vario_type_vals($ii,label) -relief flat -background gray50
	    $vario_type_table_w window configure $ii,1 -window $vario_type_tail($ii,combo)
	    $vario_type_table_w window configure $ii,2 -window $vario_type_head($ii,combo)
	    $vario_type_table_w window configure $ii,3 -window $vario_type_type($ii,combo)
	    $vario_type_table_w window configure $ii,4 -window $vario_type_threshold($ii,entry)
	}
    
    } elseif { $gamvpar(datafl) != "*.dat" } {

	### setting up the vario_type (variogram type) table
	set vario_type_table_w [table $vario_type_sframe_w.table \
				    -variable vario_type_vals \
				    -titlerows 1 \
				    -titlecols 1 \
				    -cols 5 \
				    -rows [expr $gamvpar(nvarg) + 1] \
				    -yscrollcommand "$vario_type_hsbar_w set" \
				    -height 4]
	
	$vario_type_table_w width 0 4
	$vario_type_table_w width 1 10
	$vario_type_table_w width 2 10
	$vario_type_table_w width 3 40
	$vario_type_table_w width 4 10
	
	set vario_type_vals(0,0) "\#"
	set vario_type_vals(0,1) "Tail"
	set vario_type_vals(0,2) "Head"
	set vario_type_vals(0,3) "Type"
	set vario_type_vals(0,4) "Threshold"
	
	for { set ii 1 } { $ii < [expr $gamvpar(nvarg) + 1] } { incr ii } {
	    set vario_type_vals($ii,label) [iwidgets::labeledwidget $vario_type_table_w.vals$ii \
						-labeltext "$ii" \
						-background gray50 \
						-foreground white]				       
	    

	    set vario_type_tail($ii,combo) [iwidgets::combobox $vario_type_table_w.tail$ii \
						-textvariable gamvpar(ivtail,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_tail $ii "]
	    set tmp_ivtail($ii) $gamvpar(ivtail,$ii)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if gamvpar(ivtail) is empty
	    $vario_type_tail($ii,combo) delete list 0

	    foreach i $dim_name_data {
		$vario_type_tail($ii,combo) insert list end $i
		$vario_type_tail($ii,combo) delete list [expr $gamvpar(variables) + 1]
		if { $i == $tmp_ivtail($ii) } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    $vario_type_tail($ii,combo) delete list 0
	    set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	    $vario_type_tail($ii,combo) selection set [expr $sel - 1]
	    set gamvpar(ivtail,$ii) [lindex $dim_name_data $sel]


	    set vario_type_head($ii,combo) [iwidgets::combobox $vario_type_table_w.head$ii \
						-textvariable gamvpar(ivhead,$ii) \
						-width 15 \
						-listheight 60 \
						-validate {regexpvalidate %c} \
						-selectioncommand "check_variable_head $ii"]
	    set tmp_ivhead($ii) $gamvpar(ivhead,$ii)
	    set cnt 0   ;# counter starts at 0
	    set sel 1   ;# that's the default selection if gamvpar(ivhead) is empty
	    $vario_type_head($ii,combo) delete list 0

	    foreach i $dim_name_data {
		$vario_type_head($ii,combo) insert list end $i
		$vario_type_head($ii,combo) delete list [expr $gamvpar(variables) + 1]
		if { $i == $tmp_ivhead($ii) } { ;# ok, that's the value in gamvpar, lets save the index (=cnt!) in sel
		    set sel $cnt
		}
		incr cnt
	    }
	    $vario_type_head($ii,combo) delete list 0
	    set sel [expr $sel - ($gamvpar(cancel_counter) * ($gamvpar(variables) + 1))]
	    $vario_type_head($ii,combo) selection set [expr $sel - 1]
	    set gamvpar(ivhead,$ii) [lindex $dim_name_data $sel]
	    
	    
	    set vario_type_type($ii,combo) [iwidgets::combobox $vario_type_table_w.type$ii \
						-textvariable gamvpar(ivtype,$ii) \
						-selectioncommand "threshold_activator $ii" \
						-width 40 \
						-listheight 75 \
						-validate {regexpvalidate %c}]
	    $vario_type_table_w.type$ii insert list end \
		"Traditional semi-variogram" "Traditional cross semi-variogram" "Covariance" "Correlogram" "General relative semivariogram" "Pairwise relative semi-variogram" "Semivariogram of logarithms" "Semi-madogram" "Indicator semi-variogram - continous variable" "Indicator semi-variogram - categorical variable"
	    if { $gamvpar(ivtype,$ii) == "Traditional semi-variogram" } {
		set sel 0	
	    } elseif { $gamvpar(ivtype,$ii) == "Traditional cross semi-variogram" } {
		set sel 1
	    } elseif { $gamvpar(ivtype,$ii) == "Covariance" } {
		set sel 2
	    } elseif { $gamvpar(ivtype,$ii) == "Correlogram" } {
		set sel 3
	    } elseif { $gamvpar(ivtype,$ii) == "General relative semivariogram" } {
		set sel 4
	    } elseif { $gamvpar(ivtype,$ii) == "Pairwise relative semi-variogram" } {
		set sel 5
	    } elseif { $gamvpar(ivtype,$ii) == "Semivariogram of logarithms" } {
		set sel 6
	    } elseif { $gamvpar(ivtype,$ii) == "Semi-madogram" } {
		set sel 7
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - continous variable" } {
		set sel 8
	    } elseif { $gamvpar(ivtype,$ii) == "Indicator semi-variogram - categorical variable" } {
		set sel 9
	    }
	    $vario_type_type($ii,combo) selection set $sel
	    set $vario_type_type($ii,combo) [lindex $vario_type_table_w.type$ii $sel]

	    
	    set vario_type_threshold($ii,entry) [iwidgets::entryfield $vario_type_table_w.threshold$ii \
						     -textvariable gamvpar(cut,$ii) \
						     -state disabled \
						     -textbackground gray85 \
						     -foreground gray50 \
						     -validate "check_number_vario_type_threshold %W %P %c $ii"]
	    if { $sel >= 8 } {
		$vario_type_threshold($ii,entry) configure -state normal -textbackground white -foreground black
	    }

	    
	    $vario_type_table_w window configure $ii,0 -window $vario_type_vals($ii,label) -relief flat -background gray50
	    $vario_type_table_w window configure $ii,1 -window $vario_type_tail($ii,combo)
	    $vario_type_table_w window configure $ii,2 -window $vario_type_head($ii,combo)
	    $vario_type_table_w window configure $ii,3 -window $vario_type_type($ii,combo)
	    $vario_type_table_w window configure $ii,4 -window $vario_type_threshold($ii,entry)
	}
    }

    
    pack $vario_type_hsbar_w -side right -anchor w -fill y
    pack $vario_type_table_w -side left -anchor w -padx 2 -expand 1 -fill both
    pack $vario_type_sframe_w -anchor nw -expand 1
    pack propagate $vario_type_sframe_w 0
    grid $vario_type_frame_w -row 0 -column 0 -sticky news -pady 0
    
    
    set spaceframe3_w [frame $variograms_frame_w.spaceframe3 \
			   -height 4]
    
    pack \
	$subframe1_w \
	$subframe2_w \
	$spaceframe1_w \
	$subframe3_w \
	$spaceframe2_w \
	$subframe4_w \
	$spaceframe3_w \
	-anchor w -expand 1 -padx 3 -pady 0
    
    pack \
	$variograms_iframe_w \
	-anchor nw -padx 0 -pady 0 -expand 1 -fill x
    
    pack \
	$work_frame_w \
	-anchor nw -side top -fill x -fill y -expand true \
	-padx 0 -pady 0
    
    
    $gamv_box_w.gamv_tbn view "Files"
    
    
    . configure -cursor arrow
    
    
    #############################################################################
    # Default buttons
    #############################################################################
    #
    # write_gamv_params fname [tmpflag]
    #----------------------------------------------------------------------------
    # This proc writs the parameter file for running gamv
    #
    # fname   : [<-] : name of parameter file
    # tmpflag : [<-] : optional flag (0,1). If this flag is set, the
    #                  temporary data files are created in order to
    #                  circumvent a problem with the gamv program
    #                  which forbids long filenames. The following
    #                  temporary files are created in the working
    #                  directory:
    #                    - "gamv_in.dat" (data input)
    #                    - "gamv_out.dat" (result of gamv run)
    #############################################################################
    
    set button_frame_w [frame $gamv_box_w.button_frame]
    
    set buttons_iframe_w [iwidgets::labeledframe $button_frame_w.buttons_iframe \
			      -labeltext "Defaults" \
			      -labelpos nw \
			      -borderwidth 2 \
			      -relief groove]
    set buttons_frame_w [$buttons_iframe_w childsite]; # here we have to get the actual label widget
    
    
    set open_button_w [iwidgets::pushbutton $buttons_iframe_w.open \
			   -height 27 \
			   -width 40 \
			   -defaultringpad 0 \
			   -image [image create photo -file [file join "$exedir/images/open.gif"]] \
			   -command {
			       set tmp_fname [tk_getOpenFile \
						  -initialdir [file dirname $gamvpar(datafl)] \
						  -defaultextension ".par" \
						  -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
						  -title "Select GAMV parameter file"]
			       
			       if { $tmp_fname != {} } {
				   set param_fname $tmp_fname
				   read_gamv_params $param_fname
			       }
			   }]
    
    
    set save_button_w [iwidgets::pushbutton $buttons_iframe_w.save \
			   -height 27 \
			   -width 27 \
			   -defaultringpad 0 \
			   -state disabled \
			   -image [image create photo -file [file join "$exedir/images/save.gif"]] \
			   -command {
			       set gamvpar(title) 0
			       write_gamv_params $param_fname
			   }]
    
    
    set save_as_button_w [iwidgets::pushbutton $buttons_iframe_w.save_as \
			      -height 27 \
			      -width 27 \
			      -defaultringpad 0 \
			      -image [image create photo -file [file join "$exedir/images/saveas.gif"]] \
			      -command {
				  if { ($gamvpar(datafl) != {*.dat}) } {
				      set tmp_fname [tk_getSaveFile \
							 -initialdir [file dirname $gamvpar(datafl)] \
							 -defaultextension ".par" \
							 -filetypes {{{Parameter Files} {.par}} {{All Files} {.*}}}\
							 -title "Save GAMV parameter file as ..."]
				      
				      $save_button_w configure -state normal
				  
				      if { $tmp_fname != {} } {
					  set gamvpar(title) 0
					  set param_fname $tmp_fname
					  write_gamv_params $param_fname
				      }
				  } elseif { $gamvpar(datafl) == {*.dat} } {
				      tk_messageBox \
					  -title "Defaults" \
					  -icon warning \
					  -message "You should select an input file first." \
					  -type ok
				  }
			      }]
    
    
    set cancel_button_w [iwidgets::pushbutton $buttons_iframe_w.cancel \
			     -height 27 \
			     -width 27 \
			     -defaultringpad 0 \
			     -image [image create photo -file [file join "$exedir/images/stop.gif"]] \
			     -command {
				 set private(button) 0
			     }]
    
    
    set run_button_w [iwidgets::pushbutton $buttons_iframe_w.run \
			  -height 27 \
			  -width 27 \
			  -defaultringpad 0 \
			  -image [image create photo -file [file join "$exedir/images/run.gif"]] \
			  -command {
			      if { $gamvpar(datafl) == {*.dat} } {
				  tk_messageBox \
				      -title "Defaults" \
				      -icon warning \
				      -message "You should select an input file first." \
				      -type ok
			      } elseif { ($gamvpar(datafl) != {*.dat}) } {
				  if { $tcl_platform(platform) == "unix" } {
				      set command "../gslib/bin/gamv"
				  } elseif { $tcl_platform(platform) == "windows" } {
				      set command "C:/Programme/Statios/Gslib77/bin/gamv.exe"
				  }
				  
				  set gamvpar(title) 1
				  set paramfile "tmprun.par"
				  write_gamv_params $paramfile 1
				  
				  set outf [open "GAMV.TMP" {w}]
				  puts $outf "$paramfile"
				  close $outf
				  set logfile [open "| $command < GAMV.TMP" {r}]
				  fconfigure $logfile -buffering none
				  while { ![eof $logfile] } {
				      gets $logfile input_line
				      log $input_line
				      update
				  }
				  close $logfile
				  
				  ### copy/delete the temporary files
				  catch {
				      file copy -force "gamv_out.dat" $gamvpar(outfl)
				  }
				  catch {
				      file delete -force \
					  "gamv_in.dat" \
					  "gamv_out.dat" \
					  "tmprun.par" \
					  "GAMV.TMP"
				  }
				  set private(button) 1
			      }
			  }]
    
    bind $run_button_w <Return> "
	    $run_button_w configure -state active -relief sunken
	    update idletasks
	    after 100
	    $run_button_w invoke
	"
    
    grid columnconfigure $button_frame_w 0 -minsize 15
    grid columnconfigure $button_frame_w 1 -minsize 100
    grid columnconfigure $button_frame_w 2 -minsize 15
    grid columnconfigure $button_frame_w 3 -minsize 100
    grid columnconfigure $button_frame_w 4 -minsize 15
    grid columnconfigure $button_frame_w 5 -minsize 100
    grid columnconfigure $button_frame_w 6 -minsize 15
    grid columnconfigure $button_frame_w 7 -minsize 100
    grid columnconfigure $button_frame_w 8 -minsize 15
    grid columnconfigure $button_frame_w 9 -minsize 100
    grid columnconfigure $button_frame_w 10 -minsize 15
    
    grid $open_button_w    -row 0 -column 1 -sticky we -pady 8 -padx 5
    grid $save_button_w    -row 0 -column 3 -sticky we -pady 8 -padx 5
    grid $save_as_button_w -row 0 -column 5 -sticky we -pady 8 -padx 5
    grid $cancel_button_w  -row 0 -column 7 -sticky we -pady 8 -padx 5
    grid $run_button_w     -row 0 -column 9 -sticky we -pady 8 -padx 5
    
    
    pack \
	$buttons_iframe_w \
	-padx 5 -pady 5 -expand 1
    
    
    pack $button_frame_w \
	-side top -fill x -expand true \
	-padx 0 -pady 0
    
    
    raise $gamv_box_w
    
    tkwait visibility $gamv_box_w
    tkwait visibility $run_button_w
    
    set oldFocus [focus]
    set oldGrab [grab current $gamv_box_w]
    if {$oldGrab != ""} {
	set grabStatus [grab status $oldGrab]
    }
    grab $gamv_box_w
    focus $run_button_w
    
    . configure -cursor arrow
    
    update idletask
    tkwait variable private(button)
    
    catch {focus $oldFocus}
    catch {
	# It's possible that the window has already been destroyed,
	# hence this "catch".  Delete the Destroy handler so that
	# tkPriv(button) doesn't get reset by it.
    
	bind $gamv_box_w <Destroy> {}
	destroy $gamv_box_w
    }
    
    if {$oldGrab != ""} {
	if {$grabStatus == "global"} {
	    grab -global $oldGrab
	} else {
	    grab $oldGrab
	}
    }
    return
}
#============================================================================
#=END=OF=GAMV================================================================